package it.unibo.cmdb.archimate;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.window.Window;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.progress.IProgressService;

import com.archimatetool.editor.model.IEditorModelManager;
import com.archimatetool.editor.model.IModelImporter;
import com.archimatetool.model.IArchimateModel;
import com.archimatetool.model.IFolder;
import com.archimatetool.model.ModelVersion;
import com.archimatetool.model.util.ArchimateResourceFactory;

import it.unibo.cmdb.archimate.ModelTransformer.TransformationResult;
import it.unibo.cmdb.archimate.model.transformationUtil.TransformationUtil;
import it.unibo.cmdb.archimate.model.transformationUtil.TransformationUtilFactory;
import it.unibo.cmdb.cmdbf.client.CMDBClient;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.Model;

public class CMDBImporter implements IModelImporter {	
	
	private class ImportJob implements IRunnableWithProgress {
		
		private String query;
		private IArchimateModel model;
		
		public ImportJob(String query) {
			this.query = query;
		}
		
		public IArchimateModel getModel() {
			return model;
		}

		@Override
		public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
			try {
				model = doImport(monitor, query);
			} catch (Throwable e) {
				LogUtils.logException(e);
				throw new InvocationTargetException(e, e.getMessage());
			}
		}
	}
	
	@Override
	public void doImport() throws IOException {
		CMDBDialog dialog = new CMDBDialog(Display.getCurrent().getActiveShell(), null, true);
		if (dialog.open() != Window.OK)
			return;
		
		try {
			IArchimateModel model = dialog.getModel();
			String query = dialog.getQuery();
	        
			ImportJob importJob = new ImportJob(query);
			IProgressService progressService = CMDBPlugin.INSTANCE.getWorkbench().getProgressService();
			progressService.run(true, true, importJob);
			
			IArchimateModel remoteModel = importJob.getModel();
			if(remoteModel != null) {
				remoteModel.setDefaults();
				remoteModel.setVersion(ModelVersion.VERSION);
				remoteModel.getMetadata();
				CMDBContentAdapter remoteAdapter = CMDBContentAdapter.getAdapter(remoteModel);
				remoteAdapter.saveMetadata();
				if (model == null) {
					model = remoteModel;
					if(model.eResource() != null)
						model.eResource().getContents().remove(model);
					CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(model);
					adapter.saveMetadata();
					IEditorModelManager.INSTANCE.openModel(model);
					ModelUtils.makeModelDirty(model);
				}
				else {				
					remoteModel.setId(model.getId());	
					remoteModel.setName(model.getName());	
					for(IFolder folder : remoteModel.getFolders())
						folder.setId(model.getFolder(folder.getType()).getId());		       
	
			        IEditorModelManager.INSTANCE.closeModel(model);
					CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(model);
					if(adapter != null)
						adapter.disableTrash(true);
					
					ModelMerger modelMerger = new ModelMerger(model, remoteModel);
					boolean dirty = modelMerger.merge();
					if(dirty)
						adapter.mergeMetadata(remoteModel);
					
					if(adapter != null)
						adapter.disableTrash(false);
					
					IEditorModelManager.INSTANCE.openModel(model);
					
					if(dirty)
						ModelUtils.makeModelDirty(model);
				}
			}
			
		} catch (Exception e) {
			LogUtils.logException(e);			
    		Throwable cause = null;
    		if(e instanceof InvocationTargetException && ((InvocationTargetException)e).getCause()!=null)
    			cause = ((InvocationTargetException)e).getCause();
    		else
    			cause = e;
			IStatus status = new Status(IStatus.ERROR, CMDBPlugin.PLUGIN_ID, cause.getMessage(), cause);
			ErrorDialog.openError(dialog.getShell(), this.getClass().getName(), "CMDB import aborted", status);
		}
	}
	
	private IArchimateModel doImport(IProgressMonitor monitor, String query) throws Exception {
		monitor.beginTask("Import CMDB model", 10);
		
		boolean detectDeivedRelationships = CMDBPlugin.INSTANCE.getConfiguration().isDetectDerivedRelationships();
                
        CMDBClient cmdbClient = null;
        if(!monitor.isCanceled()) {
			monitor.subTask("Initialize CMDBf client");
			cmdbClient = CMDBPlugin.INSTANCE.getCmdbClient();
			monitor.worked(1);
			if(cmdbClient == null)
				monitor.setCanceled(true);
        }

        ModelTransformer transformer = null;        
        if(!monitor.isCanceled()) {
			monitor.subTask("Initialize ATL");
			transformer = CMDBPlugin.INSTANCE.getModelTransformer();
			monitor.worked(1);
			if(transformer == null)
				monitor.setCanceled(true);
        }
        
        Model cmdbModel = null;
		if(!monitor.isCanceled()) {
			monitor.subTask("Execute CMDBf query");
	        cmdbModel = cmdbClient.query(query);
	        monitor.worked(3);
		}
		
		DiagramSerializer diagramSerializer = new DiagramSerializer();
		TransformationUtil util = TransformationUtilFactory.eINSTANCE.createTransformationUtil();
		util.setCmdbfClient(cmdbClient);
		util.setDiagramSerializer(diagramSerializer);
		IArchimateModel remoteModel = null;
        if(!monitor.isCanceled() && cmdbModel != null) {
        	monitor.subTask("Apply ATL transtormation");
        	
        	ResourceSet resSet = ArchimateResourceFactory.createResourceSet();
	        Resource resource = resSet.createResource(URI.createURI("cmdb.xmi"));
	        resource.getContents().add(cmdbModel);
	        
	        monitor.worked(1);
	        TransformationResult transformationResult = transformer.importModel(cmdbModel.eResource(), util);
			Resource outModel = transformationResult.getResource();
			
			ModelUtils.updateArchiElements(transformationResult.getTrace());
	        
	        EList<EObject> contents = outModel.getContents();
			if(contents != null) {
				for(EObject eObject : contents) {
					if(eObject instanceof IArchimateModel) {
						remoteModel = (IArchimateModel)eObject;
						remoteModel.setName(cmdbClient.getMdrId());
					}
				}
			}
	        monitor.worked(3);
		}
	        
        if(!monitor.isCanceled()) {
	        monitor.subTask("Fix Imported Model");
	        
        	ModelClosure closure = null;
	        if(detectDeivedRelationships) {
				try {
					closure = new ModelClosure(remoteModel);
				}
				catch(CycleDetectedException e) {
					LogUtils.logException(e);
				}
	        }
	        diagramSerializer.fixDiagrams(remoteModel, closure);
	        
	        monitor.worked(1);	        	        			
        }
        
		monitor.done();
		if(monitor.isCanceled())
			return null;
		else
			return remoteModel;
	}
}