package it.unibo.cmdb.archimate;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlValue;
import javax.xml.bind.annotation.adapters.XmlAdapter;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

@XmlRootElement
public class CMDBMetadata {
	
	@XmlJavaTypeAdapter(MapAdapter.class)
	private Map<String, Map<String, String>> metadata;
		
	public CMDBMetadata() {
		metadata = new HashMap<String, Map<String, String>>();
	}
	
	public Map<String, String> getMetadata(String id) {
		 return metadata.get(id);
	}
		
	public void putMetadata(String id, Map<String, String> map) {
		 metadata.put(id, map);
	}

	public String get(String id, String key) {
		Map<String, String> elementMetadata = metadata.get(id);
		return elementMetadata!=null ? elementMetadata.get(key) : null;
	}

	public void put(String id, String key, String value) {
		Map<String, String> elementMetadata = metadata.get(id);
		if(elementMetadata == null) {
			elementMetadata = new HashMap<String, String>();
			metadata.put(id, elementMetadata);
		}
		elementMetadata.put(key, value);
	}

	public void remove(String id) {
		metadata.remove(id);
	}

	public void clear() {
		metadata.clear();		
	}
	
	public void load(String content) throws JAXBException {
		StringReader reader = new StringReader(content);
		JAXBContext context = JAXBContext.newInstance(getClass());
		Unmarshaller unmarshaller = context.createUnmarshaller();
		CMDBMetadata stored = (CMDBMetadata) unmarshaller.unmarshal(reader);
		if(stored != null)
			metadata = stored.metadata;
	}
	
	public String save() throws JAXBException {
		StringWriter writer = new StringWriter();
		JAXBContext context = JAXBContext.newInstance(getClass());
		Marshaller marshaller = context.createMarshaller();
		marshaller.marshal(this, writer);
		return writer.toString();
	}
	
	@XmlRootElement
	private static class AdaptedMetadata {
		
		@XmlElement
		private List<AdaptedElementMetadata> element;
	}
	
	@XmlRootElement
	private static class AdaptedElementMetadata {
		@XmlAttribute
		private String id;
		
		@XmlElement
		private List<AdaptedMetadataProperty> property;
	}
	
	private static class AdaptedMetadataProperty {
		@XmlAttribute
		private String name;
		
		@XmlValue
		private String value;
	}
	
	private static class MapAdapter extends XmlAdapter<AdaptedMetadata, Map<String, Map<String, String>>> {

		@Override
		public Map<String, Map<String, String>> unmarshal(AdaptedMetadata adaptedMetadata) throws Exception {
			Map<String, Map<String, String>> metadata = new HashMap<String, Map<String, String>>();
			if(adaptedMetadata.element != null) {
				for(AdaptedElementMetadata element : adaptedMetadata.element) {
					Map<String, String> map = new HashMap<String, String>();
					metadata.put(element.id, map);
					if(element.property != null) {
						for(AdaptedMetadataProperty property : element.property)
							map.put(property.name, property.value);
					}
				}
			}
			return metadata;
		}

		@Override
		public AdaptedMetadata marshal(Map<String, Map<String, String>> metadata) throws Exception {
			AdaptedMetadata adaptedMetadata = new AdaptedMetadata();
			adaptedMetadata.element = new ArrayList<CMDBMetadata.AdaptedElementMetadata>();
			for(Entry<String, Map<String, String>> entry : metadata.entrySet()) {
				AdaptedElementMetadata element = new AdaptedElementMetadata();
				element.id = entry.getKey();
				adaptedMetadata.element.add(element);
				if(entry.getValue() != null) {
					element.property = new ArrayList<CMDBMetadata.AdaptedMetadataProperty>();
					for(Entry<String, String> propertyEntry : entry.getValue().entrySet()) {
						AdaptedMetadataProperty property = new AdaptedMetadataProperty();
						property.name = propertyEntry.getKey();
						property.value = propertyEntry.getValue();
						element.property.add(property);
					}
				}
			}
			return adaptedMetadata;
		}		
	}
}
