package it.unibo.cmdb.archimate;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.Map.Entry;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.ImageLoader;
import org.eclipse.swt.widgets.Display;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSSerializer;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.archimatetool.editor.diagram.util.DiagramUtils;
import com.archimatetool.model.IArchimateElement;
import com.archimatetool.model.IArchimateFactory;
import com.archimatetool.model.IArchimateModel;
import com.archimatetool.model.IArchimatePackage;
import com.archimatetool.model.IDiagramModel;
import com.archimatetool.model.IDiagramModelArchimateConnection;
import com.archimatetool.model.IDiagramModelArchimateObject;
import com.archimatetool.model.IDiagramModelConnection;
import com.archimatetool.model.IDiagramModelObject;
import com.archimatetool.model.IDiagramModelReference;
import com.archimatetool.model.IIdentifier;
import com.archimatetool.model.IRelationship;
import com.archimatetool.model.ISketchModelSticky;
import com.archimatetool.model.impl.ArchimateFactory;
import com.archimatetool.model.util.ArchimateModelUtils;

import it.unibo.cmdb.cmdbf.client.model.cmdbf.CmdbfFactory;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ItemType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.MdrScopedIdType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.Model;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ModelRecordType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.RecordType;

public class DiagramSerializer {
	
	private static final String ECLASS_TYPE_NS = "http://www.w3.org/2001/XMLSchema-instance";
	private static final String ECLASS_TYPE_ATTRIBUTE = "xsi:type";
	
	private Map<EObject, Element> xmlElementMap;
	private Map<String, Map<EObject, List<EReference>>> unresolvedReferencesMap;
	private List<IDiagramModel> diagrams;
	private List<DiagramSerialization> serializationList;
	
	private class DiagramSerialization {
		private IDiagramModel diagram;
		private ModelRecordType recordType;
		private String field;
		
		public DiagramSerialization(IDiagramModel diagram, ModelRecordType recordType, String field) {
			this.diagram = diagram;
			this.recordType = recordType;
			this.field = field;
		}

		public IDiagramModel getDiagram() {
			return diagram;
		}

		public ModelRecordType getRecordType() {
			return recordType;
		}

		public String getField() {
			return field;
		}
	}
	
	public DiagramSerializer() {
		xmlElementMap = new HashMap<EObject, Element>();
		unresolvedReferencesMap = new HashMap<String, Map<EObject,List<EReference>>>();
		diagrams = new ArrayList<IDiagramModel>();
		serializationList = new ArrayList<DiagramSerializer.DiagramSerialization>();
	}
	
	public void serialize(IDiagramModel diagram, RecordType record, String field) {
		serializationList.add(new DiagramSerialization(diagram, record.getType(), field));
	}
	
	public void deserialize(IDiagramModel diagram, RecordType record, String field) throws ParserConfigurationException, SAXException, IOException {
		String xml = (String) record.get(field);
		if(xml!=null && !xml.isEmpty())
			deserialize(xml, diagram);
	}

	public byte[] createImage(IDiagramModel diagram) {
		class CreateImageJob implements Runnable
		{
			private IDiagramModel diagram;
			private Image image;
			
			CreateImageJob(IDiagramModel diagram) {
				this.diagram = diagram;
			}			

			@Override
			public void run() {
				  image = DiagramUtils.createImage(diagram, 1, 0);
			}
			
			Image getImage() {
				return image;
			}			
			
			void dispose() {
				if(image != null)
					image.dispose();
			}
		}
		
		CreateImageJob createImageJob = new CreateImageJob(diagram);
		try{			
			Display.getDefault().syncExec(createImageJob);
			ImageLoader loader = new ImageLoader();
            loader.data = new ImageData[]{ createImageJob.getImage().getImageData() };
            ByteArrayOutputStream stream = new ByteArrayOutputStream();
            loader.save(stream, SWT.IMAGE_PNG);
            return stream.toByteArray();
		} finally {
			createImageJob.dispose();
		}
	}

	public void fixDiagrams(IArchimateModel model, ModelClosure closure) throws Exception {
		fixReferences(unresolvedReferencesMap, model);
		for(IDiagramModel diagram : model.getDiagramModels())
			fixDiagram(xmlElementMap, diagram, closure);
	}
	
	public boolean updateDiagrams(Model model) throws Exception {
		if(!serializationList.isEmpty()) {
			for(DiagramSerialization diagramSerialization : serializationList) {
				IDiagramModel diagram = diagramSerialization.getDiagram();
				String xml = serialize(diagram);
				ItemType item = model.addItem();
				RecordType record = item.addRecord(diagramSerialization.getRecordType().getNamespace(), diagramSerialization.getRecordType().getLocalName());
				record.set(diagramSerialization.getField(), xml);
				CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(diagram);
				if(adapter != null) {
					String cmdbMdrId = adapter.getMdrId(diagram);
					String cmdbLocalId = adapter.getLocalId(diagram);
					if(cmdbMdrId != null && cmdbLocalId!=null)
						item.addInstanceId(cmdbMdrId, cmdbLocalId);
				}
				String archiMdrId = ModelUtils.getArchiMdrId(diagram);
				String archiLocalId = ModelUtils.getArchiLocalId(diagram);
				if(archiMdrId != null && archiLocalId!=null)
					item.addInstanceId(archiMdrId, archiLocalId);
			}
			serializationList.clear();
			return true;
		}
		else
			return false;
	}
	
	private String serialize(IDiagramModel diagram) throws Exception {
		DocumentBuilderFactory builderFactory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = builderFactory.newDocumentBuilder();
		Document document = builder.newDocument();
		DOMImplementationLS domImplementation = (DOMImplementationLS) document.getImplementation();
		LSSerializer lsSerializer = domImplementation.createLSSerializer();
		
		Element root = document.createElement(diagram.eClass().getName());
		document.appendChild(root);
		FeatureFilter featureFilter = new FeatureFilter();
		featureFilter.add(IArchimatePackage.eINSTANCE.getDiagramModel().getEStructuralFeature(IArchimatePackage.DIAGRAM_MODEL__PROPERTIES));
		serializeObject(diagram, root, featureFilter, diagram);
		
		return lsSerializer.writeToString(document);
	}

	private IDiagramModel deserialize(String value, IDiagramModel diagram) throws ParserConfigurationException, SAXException, IOException {
		DocumentBuilderFactory builderFactory = DocumentBuilderFactory.newInstance();			
		StringReader reader = new StringReader(value);
		DocumentBuilder builder = builderFactory.newDocumentBuilder();
		Document document = builder.parse(new InputSource(reader));
		FeatureFilter featureFilter = new FeatureFilter();
		featureFilter.add(IArchimatePackage.eINSTANCE.getDiagramModel().getEStructuralFeature(IArchimatePackage.DIAGRAM_MODEL__PROPERTIES));
		featureFilter.add(IArchimatePackage.eINSTANCE.getDiagramModelArchimateObject().getEStructuralFeature(IArchimatePackage.DIAGRAM_MODEL_ARCHIMATE_OBJECT__NAME));
		featureFilter.add(IArchimatePackage.eINSTANCE.getDiagramModelArchimateConnection().getEStructuralFeature(IArchimatePackage.DIAGRAM_MODEL_ARCHIMATE_CONNECTION__NAME));
		deserializeObject(diagram, document.getDocumentElement(), featureFilter, xmlElementMap, unresolvedReferencesMap);
		diagrams.add(diagram);
		return diagram;
	}
	
	@SuppressWarnings("rawtypes")
	private void serializeObject(EObject eObject, Element xmlElement, FeatureFilter featureFilter, EObject container) throws Exception {
		xmlElement.setAttributeNS(ECLASS_TYPE_NS, ECLASS_TYPE_ATTRIBUTE, eObject.eClass().getName());			
		for(EStructuralFeature feature : eObject.eClass().getEAllStructuralFeatures()){
			if(feature.isChangeable() && (featureFilter==null || !featureFilter.contains(eObject.eClass(), feature))) {
				if(feature instanceof EReference){
					EReference eReference = (EReference)feature;
					if(eReference.isContainment()){
						if(eReference.isMany()) {
							for(Object item : ((EList)eObject.eGet(eReference))) {	
								Element xmlChild = xmlElement.getOwnerDocument().createElement(eReference.getName());
								xmlElement.appendChild(xmlChild);
								serializeObject((EObject)item, xmlChild, featureFilter, container);
							}
						}
						else {
							EObject value = (EObject)eObject.eGet(eReference);
							if(value != null) {
								Element xmlChild = xmlElement.getOwnerDocument().createElement(eReference.getName());
								xmlElement.appendChild(xmlChild);
								serializeObject(value, xmlChild, featureFilter, container);
							}
						}
					}
					else{
						if(eReference.isMany()) {
							StringBuffer xmlValue = new StringBuffer();
							for(Object item : ((EList)eObject.eGet(eReference))) {	
								if(xmlValue.length() > 0)
									xmlValue.append(' ');
								xmlValue.append(serializeReference((EObject)item));
								xmlElement.setAttribute(eReference.getName(), xmlValue.toString());
							}
						}
						else {
							EObject value = (EObject)eObject.eGet(eReference);
							if(value != null) {
								String xmlValue = serializeReference(value);									
								xmlElement.setAttribute(eReference.getName(), xmlValue);
							}
						}
					}
				}
				else if(feature instanceof EAttribute){
					EAttribute eAttribute = (EAttribute)feature;
					if(eAttribute.isMany()) {
						for(Object item : ((EList)eObject.eGet(eAttribute))) {	
							Element xmlChild = xmlElement.getOwnerDocument().createElement(eAttribute.getName());							
							String xmlValue = serializeAttribute(item, eAttribute);																			
							xmlChild.setTextContent(xmlValue);
						}
					}
					else {
						Object value = eObject.eGet(eAttribute);
						if(value != null) {
							String xmlValue = serializeAttribute(value, eAttribute);									
							xmlElement.setAttribute(eAttribute.getName(), xmlValue);
						}
					}
				}
			}
		}
	}
	
	private String serializeReference(EObject value) throws Exception{
		EAttribute IDAttribute = value.eClass().getEIDAttribute(); 
		if(IDAttribute != null) {
			Object objectId = value.eGet(IDAttribute);
			if(objectId instanceof String) {
				try{
					UUID.fromString((String) objectId);
				} catch(Exception e) {
					value.eSet(IDAttribute, UUID.randomUUID().toString());
				}
			}
		}
		
		String id = null;
		if(value instanceof IIdentifier) {
			IIdentifier element = (IIdentifier)value;
			CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(value);
			if(adapter != null) {
				String mdrId = adapter.getMdrId(element);
				String localId = adapter.getLocalId(element);
				if(mdrId==null && localId==null) {
					mdrId = ModelUtils.getArchiMdrId(element);
					localId = ModelUtils.getArchiLocalId(element);				
				}
				if(mdrId!=null && localId!=null) {
					MdrScopedIdType instanceId = CmdbfFactory.eINSTANCE.createMdrScopedIdType();
					instanceId.setMdrId(mdrId);
					instanceId.setLocalId(localId);
					id = instanceId.toString();
				}
			}
		}
		if(id == null)
			id = value.eResource().getURIFragment(value);
		return id;
	}
	
	private String serializeAttribute(Object value, EAttribute feature){
		String strValue = null;
		EDataType eType = (EDataType)feature.getEType();
		EFactory factory = eType.getEPackage().getEFactoryInstance();
	    strValue = factory.convertToString((EDataType)eType, value);		
		return strValue;
	}
	
	@SuppressWarnings("unchecked")
	private void deserializeObject(EObject eObject, Element xmlElement, FeatureFilter featureFilter, Map<EObject, Element> xmlMap, Map<String, Map<EObject, List<EReference>>> references) {
		xmlMap.put(eObject, xmlElement);
		if (xmlElement.hasAttributes()) {
			for (int i = 0; i < xmlElement.getAttributes().getLength(); i++) {
				Attr attribute = (Attr) xmlElement.getAttributes().item(i);
				if (attribute.getValue() != null) {
					EStructuralFeature feature = eObject.eClass().getEStructuralFeature(attribute.getName());
					if(feature!=null && feature.isChangeable() && (featureFilter==null || !featureFilter.contains(eObject.eClass(), feature))) {
						if (feature instanceof EReference) {
							EReference reference = (EReference)feature;
							if (feature.isMany()) {
								for (String value : attribute.getValue().split(" ")) {
									EObject target = deserializeReference(value, eObject.eResource(), null);
									if (target != null)
										((EList<EObject>) eObject.eGet(reference)).add(target);
									else
										addUnresolvedReference(references, value, eObject, reference);
								}
							} else {
								String value = attribute.getValue();
								EObject target = deserializeReference(value, eObject.eResource(), null);
								if (target != null)
									eObject.eSet(reference, target);
								else
									addUnresolvedReference(references, value, eObject, reference);
							}
						} else if (feature instanceof EAttribute) {
							Object value = deserializeAttribute(attribute.getValue(), (EAttribute) feature);
							eObject.eSet(feature, value);
						}
					}
				}
			}
		}
		if (xmlElement.hasChildNodes()) {
			for (int i = 0; i < xmlElement.getChildNodes().getLength(); i++) {
				Node xmlNode = (Node) xmlElement.getChildNodes().item(i);
				if (xmlNode instanceof Element) {
					Element xmlChild = (Element) xmlNode;
					EStructuralFeature feature = eObject.eClass().getEStructuralFeature(xmlChild.getNodeName());
					if(feature!=null && feature.isChangeable() && (featureFilter==null || !featureFilter.contains(eObject.eClass(), feature))) {
						if (feature instanceof EReference) {
							String childType = xmlChild.getAttribute(ECLASS_TYPE_ATTRIBUTE);
							EClass childEClass = (EClass) IArchimateFactory.eINSTANCE.getArchimatePackage().getEClassifier(childType);
							EObject child = IArchimateFactory.eINSTANCE.create(childEClass);
							if (feature.isMany())
								((EList<EObject>) eObject.eGet(feature)).add(child);
							else
								eObject.eSet(feature, child);
							deserializeObject(child, xmlChild, featureFilter, xmlMap, references);
						} else if (feature instanceof EAttribute) {
							Object value = deserializeAttribute(xmlChild.getTextContent(), (EAttribute) feature);
							if (feature.isMany())
								((EList<Object>) eObject.eGet(feature)).add(value);
							else
								eObject.eSet(feature, value);
						}
					}
				}
			}
		}
	}
	
	private void addUnresolvedReference(Map<String, Map<EObject, List<EReference>>> references, String id, EObject eObject, EReference reference) {
		 Map<EObject, List<EReference>> referenceMap = references.get(id);
		 if(referenceMap == null) {
			 referenceMap = new HashMap<EObject, List<EReference>>();
			 references.put(id, referenceMap);
		 }
		 List<EReference> list = referenceMap.get(eObject);
       if (list == null) {
               list = new ArrayList<EReference>();
               referenceMap.put(eObject, list);
       }
       list.add(reference);
	}

	private EObject deserializeReference(String value, Resource resource, Map<String, EObject>idMap) {
		EObject eObject = null;
		if(idMap != null)
			eObject = idMap.get(value);
		if(eObject == null)
			eObject = resource.getEObject(value);
		return eObject;
	}

	private Object deserializeAttribute(String value, EAttribute feature) {
		EDataType eType = (EDataType) feature.getEType();
		EFactory factory = eType.getEPackage().getEFactoryInstance();
		return factory.createFromString((EDataType) eType, value);
	}

	@SuppressWarnings("unchecked")
	private void fixReferences(Map<String, Map<EObject, List<EReference>>> references, IArchimateModel model) throws Exception {
		Map<String, EObject> idMap = buildIdMap(model);
		
		Iterator<Entry<String, Map<EObject, List<EReference>>>> mapIterator = references.entrySet().iterator();
		while(mapIterator.hasNext()) {
			Entry<String, Map<EObject, List<EReference>>> mapEntry = mapIterator.next();
			if(!mapEntry.getValue().isEmpty()) {
				EObject reference = deserializeReference(mapEntry.getKey(), model.eResource(), idMap);		
				if (reference != null) {
					for (Entry<EObject, List<EReference>> referenceEntry : mapEntry.getValue().entrySet()) {				
			            for(EReference feature : referenceEntry.getValue()) {
		                    if (feature.isMany())
		                        ((EList<EObject>) referenceEntry.getKey().eGet(feature)).add(reference);
		                    else
		                    	referenceEntry.getKey().eSet(feature, reference);
		            	}
		            }
					mapIterator.remove();
				}
	            else
	            	LogUtils.logWarning("deserializeReference: " + mapEntry.getKey() + " NOT FOUND");
			}
		}		
	}
	
	private void fixDiagram(Map<EObject, Element> xmlMap, IDiagramModel diagram, ModelClosure closure) {
		CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(diagram);
		
		List<EObject> deleteList = new ArrayList<EObject>();
		TreeIterator<EObject> contentIterator = diagram.eAllContents();
		while(contentIterator.hasNext()) {
			EObject eObject = contentIterator.next();
			if (eObject instanceof IDiagramModelArchimateObject) {
				IDiagramModelArchimateObject archimateObject = (IDiagramModelArchimateObject) eObject; 
				if (archimateObject.getArchimateElement() == null || adapter.isDeleted(archimateObject.getArchimateElement()))
					deleteList.add(eObject);					
			} else if (eObject instanceof IDiagramModelArchimateConnection) {
				IDiagramModelArchimateConnection connection = (IDiagramModelArchimateConnection) eObject;
				IArchimateElement sourceElement = null;
				IDiagramModelObject source = connection.getSource();
				if (source != null && source instanceof IDiagramModelArchimateObject) {
					sourceElement = ((IDiagramModelArchimateObject) source).getArchimateElement();
					if(sourceElement != null && adapter.isDeleted(sourceElement))
						sourceElement = null;
				}
				IArchimateElement targetElement = null;
				IDiagramModelObject target = connection.getTarget();
				if (target != null && target instanceof IDiagramModelArchimateObject) {
					targetElement = ((IDiagramModelArchimateObject) target).getArchimateElement();
					if(targetElement != null && adapter.isDeleted(targetElement))
						targetElement = null;
				}
				IRelationship connectionRelationship = connection.getRelationship();
				if(connectionRelationship!=null && adapter.isDeleted(connectionRelationship))
					connectionRelationship = null;
				if (connectionRelationship==null && sourceElement != null && targetElement != null) {
					EClass pathType = closure!=null ? closure.getPath(sourceElement, targetElement) : null;
					if(pathType != null) {					
						for (IRelationship relationship : ArchimateModelUtils.getSourceRelationships(sourceElement)) {
							if (relationship.getTarget() == targetElement && relationship.eClass() == pathType && !adapter.isDeleted(relationship)) {
								connection.setRelationship(relationship);
								connectionRelationship = connection.getRelationship();
							}
						}
						if(connection.getRelationship()==null) {
							IRelationship relationship = (IRelationship)IArchimateFactory.eINSTANCE.create(pathType);
							relationship.setSource(sourceElement);
							relationship.setTarget(targetElement);
							connection.setRelationship(relationship);
							connection.addRelationshipToModel(adapter.getDerivedFolder());
							connection.reconnect();
							connectionRelationship = connection.getRelationship();
						}
					}
				}
				if (connectionRelationship==null || sourceElement==null || targetElement==null)
					deleteList.add(connection);
				else
					connection.reconnect();
			} else if (eObject instanceof IDiagramModelReference) {
				IDiagramModelReference modelReference = (IDiagramModelReference) eObject;
				if (modelReference.getReferencedModel() == null || adapter.isDeleted(modelReference))
					deleteList.add(modelReference);
			}
		}

		Map<EObject, EObject> deleteMap = new HashMap<EObject, EObject>();
		for (EObject eObject : deleteList) {
			Element xmlElement = xmlMap!=null ? xmlMap.get(eObject) : null;
			String name = xmlElement != null ? xmlElement.getAttribute("name") : null;
			if (eObject instanceof IDiagramModelArchimateObject) {
				IDiagramModelArchimateObject diagramObj = (IDiagramModelArchimateObject)eObject;
				if(diagramObj.getArchimateElement() != null)
					name = diagramObj.getArchimateElement().getName();
				ISketchModelSticky sketck = ArchimateFactory.eINSTANCE.createSketchModelSticky();
				sketck.setName(name);
				sketck.setContent(name);
				sketck.setFillColor("#FF0000");
				sketck.setFontColor("#FFFF00");
				sketck.setFont(diagramObj.getFont());
				sketck.setBounds(diagramObj.getBounds());
				sketck.setTextAlignment(diagramObj.getTextAlignment());
				//sketck.setTextPosition(diagramObj.getTextPosition());
				diagram.getChildren().add(sketck);
				deleteMap.put(eObject, sketck);
			} else if (eObject instanceof IDiagramModelArchimateConnection) {
				IDiagramModelArchimateConnection connection = (IDiagramModelArchimateConnection) eObject;
				if(connection.getRelationship() != null)
					name = connection.getRelationship().getName();
				IDiagramModelConnection diagramConn = ArchimateFactory.eINSTANCE.createDiagramModelConnection();
				diagramConn.setLineColor("#FF0000");
				diagramConn.setFontColor("#FF0000");
				diagramConn.setDocumentation(connection.getDocumentation());
				diagramConn.setName(name);
				diagramConn.setFont(connection.getFont());
				diagramConn.setLineWidth(connection.getLineWidth());
				diagramConn.setTextPosition(connection.getTextPosition());
				diagramConn.setType(connection.getType());
				//diagramConn.setTextAlignment(connection.getTextAlignment());
				diagramConn.getBendpoints().addAll(connection.getBendpoints());
				deleteMap.put(eObject, diagramConn);
			} else if (eObject instanceof IDiagramModelReference) {
				IDiagramModelReference modelReference = (IDiagramModelReference)eObject;
				if(modelReference.getReferencedModel() != null)
					name = modelReference.getReferencedModel().getName();
				ISketchModelSticky sketck = ArchimateFactory.eINSTANCE.createSketchModelSticky();
				sketck.setName(name);
				sketck.setContent(name);
				sketck.setFillColor("#FF0000");
				sketck.setFontColor("#000000");
				sketck.setFont(modelReference.getFont());
				sketck.setBounds(modelReference.getBounds());
				sketck.setTextAlignment(modelReference.getTextAlignment());
				//sketck.setTextPosition(modelReference.getTextPosition());
				diagram.getChildren().add(sketck);
				deleteMap.put(eObject, sketck);
			}
		}
		
		for (Entry<EObject, EObject> deletedEntry : deleteMap.entrySet()) {
			if(deletedEntry.getKey() instanceof IDiagramModelConnection && deletedEntry.getValue() instanceof IDiagramModelConnection) {
				IDiagramModelConnection deletedConn = (IDiagramModelConnection)deletedEntry.getKey();
				IDiagramModelConnection diagramConn = (IDiagramModelConnection)deletedEntry.getValue();
				IDiagramModelObject source = deletedConn.getSource();
				IDiagramModelObject target = deletedConn.getTarget();
				if(source != null && deleteMap.containsKey(source))
					source = (IDiagramModelObject)deleteMap.get(source);
				if(target != null && deleteMap.containsKey(target))
					target = (IDiagramModelObject)deleteMap.get(target);
				if(source!=null && target!=null) {
					diagramConn.setSource(source);
					diagramConn.setTarget(target);
					diagramConn.reconnect();
				}
			}			
		}
		
		for (EObject deleted : deleteList) {
			if(deleted instanceof IDiagramModelObject)
				diagram.getChildren().remove(deleted);
			else if(deleted instanceof IDiagramModelConnection)
				((IDiagramModelConnection)deleted).disconnect();			
		}
	}
	
	private Map<String, EObject> buildIdMap(EObject container) throws Exception {
		Map<String, EObject> idMap = new HashMap<String, EObject>();
		TreeIterator<EObject> iterator = container.eAllContents();
		while(iterator.hasNext()) {
			EObject eObject = iterator.next();
			if(eObject instanceof IIdentifier) {
				IIdentifier element = (IIdentifier)eObject;
				CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(element);
				if(adapter != null)
				{
					String cmdbMdrId = adapter.getMdrId(element);
					String cmdbLocalId = adapter.getLocalId(element);
					if(cmdbMdrId!=null && cmdbLocalId!=null)			
						addId(cmdbMdrId, cmdbLocalId, element, idMap);
					
					String archiMdrId = ModelUtils.getArchiMdrId(element);
					String archiLocalId = ModelUtils.getArchiLocalId(element);
					if(archiMdrId!=null && archiLocalId!=null)			
						addId(archiMdrId, archiLocalId, element, idMap);				
				}
			}
		}
		return idMap;
	}
	
	private void addId(String mdrId, String localId, EObject eObject, Map<String, EObject> idMap) {
		MdrScopedIdType instanceId = CmdbfFactory.eINSTANCE.createMdrScopedIdType();
		instanceId.setMdrId(mdrId);
		instanceId.setLocalId(localId);
		String id = instanceId.toString();
		idMap.put(id, eObject);
	}
	
	private static class FeatureFilter {
		Map<EClass, Set<EStructuralFeature>> featureFilter;
		
		private FeatureFilter() {
			featureFilter = new HashMap<EClass, Set<EStructuralFeature>>();
		}
		
		public void add(EStructuralFeature feature) {
			Set<EStructuralFeature> featureSet = featureFilter.get(feature.getEContainingClass());
			if(featureSet == null) {
				featureSet = new HashSet<EStructuralFeature>();
				featureFilter.put(feature.getEContainingClass(), featureSet);				
			}
			featureSet.add(feature);				
		}
		
		public boolean contains(EClass type, EStructuralFeature feature) {
			Set<EStructuralFeature> featureSet = featureFilter.get(type);
			boolean ok = featureSet!=null && featureSet.contains(feature);
			for(EClass superType : type.getESuperTypes())
				ok |= contains(superType, feature);
			return ok;
		}
	}
}
