package it.unibo.cmdb.archimate;

import java.lang.reflect.InvocationTargetException;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.xml.datatype.XMLGregorianCalendar;

import org.eclipse.compare.CompareUI;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.diffmerge.api.IMatch;
import org.eclipse.emf.diffmerge.api.Role;
import org.eclipse.emf.diffmerge.api.scopes.IEditableModelScope;
import org.eclipse.emf.diffmerge.api.scopes.IFeaturedModelScope;
import org.eclipse.emf.diffmerge.api.scopes.IModelScope;
import org.eclipse.emf.diffmerge.impl.policies.DefaultDiffPolicy;
import org.eclipse.emf.diffmerge.impl.policies.DefaultMatchPolicy;
import org.eclipse.emf.diffmerge.impl.policies.DefaultMergePolicy;
import org.eclipse.emf.diffmerge.impl.scopes.SubtreeModelScope;
import org.eclipse.emf.diffmerge.ui.setup.EMFDiffMergeEditorInput;
import org.eclipse.emf.diffmerge.ui.specification.AbstractScopeDefinition;
import org.eclipse.emf.diffmerge.ui.specification.IComparisonMethod;
import org.eclipse.emf.diffmerge.ui.specification.IModelScopeDefinition;
import org.eclipse.emf.diffmerge.ui.specification.ext.DefaultComparisonMethod;
import org.eclipse.emf.diffmerge.ui.util.DiffMergeLabelProvider;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.jface.viewers.ILabelProvider;

import com.archimatetool.model.IArchimateElement;
import com.archimatetool.model.IArchimateModel;
import com.archimatetool.model.IArchimatePackage;
import com.archimatetool.model.IBounds;
import com.archimatetool.model.IDiagramModel;
import com.archimatetool.model.IDiagramModelArchimateConnection;
import com.archimatetool.model.IDiagramModelArchimateObject;
import com.archimatetool.model.IDiagramModelBendpoint;
import com.archimatetool.model.IDiagramModelComponent;
import com.archimatetool.model.IDiagramModelConnection;
import com.archimatetool.model.IDiagramModelReference;
import com.archimatetool.model.IFolder;
import com.archimatetool.model.IIdentifier;
import com.archimatetool.model.IMetadata;
import com.archimatetool.model.INameable;
import com.archimatetool.model.IProperty;
import com.archimatetool.model.IRelationship;

public class ModelMerger {
	
	private IModelScopeDefinition localModelScopeDefinition;
	private IModelScopeDefinition remoteModelScopeDefinition;
	private Map<EObject, Set<EObject>> localDependentsMap;
		
	public ModelMerger(IArchimateModel local, IArchimateModel remote) throws InvocationTargetException, InterruptedException {
		localModelScopeDefinition = new ArchiScopeDefinition(local, "Archi", true);
		remoteModelScopeDefinition = new ArchiScopeDefinition(remote, "CMDB", false);
		localDependentsMap = buildDependentsMap(local);
	}

	public boolean merge() {
		IComparisonMethod method = new DefaultComparisonMethod(remoteModelScopeDefinition, localModelScopeDefinition, null) {
			@Override
			protected org.eclipse.emf.diffmerge.api.IMatchPolicy createMatchPolicy() {
				return new ArchiMatchPolicy();
			}
			
			@Override
			protected org.eclipse.emf.diffmerge.api.IDiffPolicy createDiffPolicy() {
				return new ArchiDiffPolicy();
			}
			
			@Override
			protected org.eclipse.emf.diffmerge.api.IMergePolicy createMergePolicy() {
				return new ArchiMergePolicy();
			}
			
			@Override
			protected ILabelProvider getCustomLabelProvider() {
				return new ArchiLabelProvider();
			}
		};
		
        ArchiEMFDiffMergeEditorInput input = new ArchiEMFDiffMergeEditorInput(method);
        CompareUI.openCompareDialog(input);
        return input.isOkPressed();
	}
	
	private Map<EObject, Set<EObject>> buildDependentsMap(IArchimateModel model) {
		Map<EObject, Set<EObject>> dependentsMap = new HashMap<EObject, Set<EObject>>();
		TreeIterator<EObject> iterator = model.eAllContents();
		while(iterator.hasNext()) {
			EObject eObject = iterator.next();
			if(eObject instanceof IRelationship) {
				IRelationship relationship = (IRelationship)eObject;
				addDependent(dependentsMap, relationship.getSource(), relationship);
				addDependent(dependentsMap, relationship.getTarget(), relationship);				
			}
			if(eObject instanceof IDiagramModelReference) {
				IDiagramModelReference diagramModelReference = (IDiagramModelReference)eObject;
				addDependent(dependentsMap, diagramModelReference.getReferencedModel(), diagramModelReference);
			}
			if(eObject instanceof IDiagramModelArchimateObject) {
				IDiagramModelArchimateObject diagramModelArchimateObject = (IDiagramModelArchimateObject)eObject;
				addDependent(dependentsMap, diagramModelArchimateObject.getArchimateElement(), diagramModelArchimateObject);				
			}
			if(eObject instanceof IDiagramModelArchimateConnection) {
				IDiagramModelArchimateConnection diagramModelArchimateConnection = (IDiagramModelArchimateConnection)eObject;
				addDependent(dependentsMap, diagramModelArchimateConnection.getRelationship(), diagramModelArchimateConnection);
			}
			if(eObject instanceof IDiagramModelConnection) {
				IDiagramModelConnection diagramModelConnection = (IDiagramModelConnection)eObject;
				addDependent(dependentsMap, diagramModelConnection.getSource(), diagramModelConnection);				
				addDependent(dependentsMap, diagramModelConnection.getTarget(), diagramModelConnection);
			}
		}
		return dependentsMap;
	}
	
	private void addDependent(Map<EObject, Set<EObject>> dependentsMap, EObject eObject, EObject dependent) {
		Set<EObject> dependentsSet = dependentsMap.get(eObject);
		if(dependentsSet == null) {
			dependentsSet = new HashSet<EObject>();
			dependentsMap.put(eObject, dependentsSet);
		}
		dependentsSet.add(dependent);
	}
	
	private class ArchiMatchID implements Comparable<ArchiMatchID>{
		private String cmdbMdrId;
		private String cmdbLocalId;
		private String archiMdrId;
		private String archiLocalId;
		private String component;
		
		public ArchiMatchID(IArchimateElement element, String component) {
			this.component = component;
			
			CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(element);
			cmdbMdrId = adapter.getMdrId(element);
			cmdbLocalId = adapter.getLocalId(element);
			archiMdrId = ModelUtils.getArchiMdrId(element);
			try {
				archiLocalId = ModelUtils.getArchiLocalId(element);
			} catch (Exception e) {
				archiLocalId = null;
				IStatus status = new Status(IStatus.ERROR, CMDBPlugin.PLUGIN_ID, e.getMessage(), e);
				CMDBPlugin.INSTANCE.getLog().log(status);
			}
			
			if(cmdbLocalId == null && archiLocalId == null) {
				IStatus status = new Status(IStatus.ERROR, CMDBPlugin.PLUGIN_ID, "ArchiMatchID is NULL:" + ModelUtils.getLabel(element));
				CMDBPlugin.INSTANCE.getLog().log(status);
			}
		}

		@Override
		public boolean equals(Object obj) {
			boolean equals = false;
			if(obj instanceof ArchiMatchID){
				ArchiMatchID id = (ArchiMatchID)obj;
				if(cmdbLocalId != null && id.cmdbLocalId != null)					
					equals = equals(cmdbMdrId, cmdbLocalId, id.cmdbMdrId, id.cmdbLocalId);
				else
					equals = true;
				equals &= equals(archiMdrId, archiLocalId, id.archiMdrId, id.archiLocalId);
				if(component == null)
					equals &= id.component == null;
				else
					equals &= component.equals(id.component);
			}
			return equals;
	    }
		
		@Override
		public int compareTo(ArchiMatchID id) {
			boolean comparated = false;
			int compare = 0;
			if(cmdbLocalId != null && id.cmdbLocalId != null) {
				compare = compareTo(cmdbMdrId, cmdbLocalId, id.cmdbMdrId, id.cmdbLocalId);
				comparated = true;
			}
			if((!comparated || compare != 0) && archiLocalId != null && id.archiLocalId != null) {
				compare = compareTo(archiMdrId, archiLocalId, id.archiMdrId, id.archiLocalId);
				comparated = true;
			}
			if(comparated && compare == 0) {
				if(component == null)
					compare = id.component==null ? 0 : -1;
				else
					compare = id.component==null ? 1 : component.compareTo(id.component);
			}
			if(comparated)
				return compare;
			else
				throw new NullPointerException("ArchiMatchId is NULL");
		}
				
		private boolean equals(String mdrId1, String localId1, String mdrId2, String localId2) {
			boolean equals = false;
			if(mdrId1 != null && localId1 != null)
				equals = mdrId1.equals(mdrId2) && localId1.equals(localId2);
			return equals;
		}
		
		private int compareTo(String mdrId1, String localId1, String mdrId2, String localId2) {
			int compare = 0;
			if(mdrId1 != null && mdrId2 !=null )
				compare = mdrId1.compareTo(mdrId2);
			else 
				throw new NullPointerException("mdrId is null");
			if(compare == 0) {
				if(localId1 != null && localId2 != null)
					compare = localId1.compareTo(localId2);
				else 
					throw new NullPointerException("localId is null");
			}
			return compare;
		}
		
		@Override
		public String toString() {
			StringBuilder builder = new StringBuilder();
			builder.append("[CMDB:{");
			builder.append(cmdbMdrId);
			builder.append(", ");
			builder.append(cmdbLocalId);
			builder.append("} Archi:{");
			builder.append(archiMdrId);
			builder.append(", ");
			builder.append(archiLocalId);
			builder.append("}]");
			return builder.toString();
		}
	}
	
	private class ArchiMatchPolicy extends DefaultMatchPolicy {

		@Override
		public Object getMatchID(EObject element, IModelScope scope) {
			Object id = null;
			if(element instanceof IArchimateModel)
				id = element.eClass().getName();
			else if(element instanceof IFolder)
				id = element.eClass().getName() + "-" + getFolderId((IFolder) element);
			else if (element instanceof IArchimateElement)
				id = new ArchiMatchID((IArchimateElement) element, null);
			else if(element.eContainer() instanceof IArchimateElement)
				id = new ArchiMatchID((IArchimateElement) element.eContainer(), getComponentName(element));
			else if (element instanceof IIdentifier)
				id = ((IIdentifier) element).getId();
			else if(element.eContainer() instanceof IArchimateElement)
				id = ((IIdentifier)element.eContainer()).getId() + "-" + getComponentName(element);
			return id;
		}
		
		@Override
		public java.util.Comparator<Object> getMatchIDComparator() {
			return new Comparator<Object>() {

				@SuppressWarnings({ "rawtypes", "unchecked" })
				@Override
				public int compare(Object o1, Object o2) {
					if(o1.getClass().equals(o2.getClass()))
						return ((Comparable)o1).compareTo(o2);
					else
						return o1.getClass().getName().compareTo(o2.getClass().getName());
				}				
			};
		}
		
		private String getComponentName(EObject eObject) {
			String component = null;
			EObject container = eObject.eContainer(); 
			if(container != null) {
				EStructuralFeature containingFeature = eObject.eContainingFeature();
				if(containingFeature.isMany()) {
					@SuppressWarnings("unchecked")
					EList<EObject> content = (EList<EObject>)container.eGet(containingFeature, false);
					component = containingFeature.getName() + "-" + Integer.toString(content.indexOf(eObject));
				}
			}
			return component;
		}
		
	    private String getFolderId(IFolder folder){
	    	StringBuilder id = new StringBuilder();
	    	for(EObject eObject = folder; eObject instanceof IFolder; eObject = eObject.eContainer()) {
	    		if(id.length()>0)
	    			id.insert(0, "/");
	    		id.insert(0,((INameable) eObject).getName().replace("/", "\\/"));
	    	}
	    	return id.toString();
	    }			
	}
	
	private class ArchiDiffPolicy extends DefaultDiffPolicy {

		@Override
		public boolean considerOrdered(EStructuralFeature feature) {
			return false;
		}
		
		@Override
		public boolean coverMatch(IMatch match) {
			boolean cover = super.coverMatch(match);
			
			if(cover) {
				EObject target = match.get(Role.TARGET);
				EObject reference = match.get(Role.REFERENCE);
				if(target instanceof IMetadata || reference instanceof IMetadata)
					cover = false;
				else if(target instanceof IDiagramModelComponent && !(target instanceof IDiagramModel)) {
					IDiagramModel diagram = ((IDiagramModelComponent)target).getDiagramModel();
					if(diagram != null)
						cover = coverMatch(match.getMapping().getMatchFor(diagram, Role.TARGET));
				}
				else if(reference instanceof IDiagramModelComponent && !(reference instanceof IDiagramModel)) {
					IDiagramModel diagram = ((IDiagramModelComponent)reference).getDiagramModel();
					if(diagram != null)
						cover = coverMatch(match.getMapping().getMatchFor(diagram, Role.REFERENCE));
				}
				else if(reference instanceof IProperty || reference instanceof IBounds || reference instanceof IDiagramModelBendpoint) {
					EObject container = reference.eContainer(); 
					if(container != null)
						cover = coverMatch(match.getMapping().getMatchFor(container, Role.REFERENCE));
				}
				else if(target instanceof IProperty || target instanceof IBounds || target instanceof IDiagramModelBendpoint) {
					EObject container = target.eContainer(); 
					if(container != null)
						cover = coverMatch(match.getMapping().getMatchFor(container, Role.TARGET));
				}
				else {
					XMLGregorianCalendar targetVersion = null;
					XMLGregorianCalendar referenceVersion = null;
					String referenceId = null;
					
					if(target instanceof IIdentifier) {
						CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(target);
						targetVersion = adapter.getLastModified((IIdentifier)target);
					}
					if(reference instanceof IIdentifier) {
						CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(reference);
						referenceVersion = adapter.getLastModified((IIdentifier)reference);
						referenceId = adapter.getLocalId((IIdentifier)reference);
					}
															
					if(referenceVersion!=null && targetVersion!=null && referenceVersion.compare(targetVersion)>=0)
						cover = false; // Archi newer version 
					else if(target==null && referenceId==null)
						cover = false; // Archi new element
					else if(reference!=null && CMDBContentAdapter.getAdapter(reference).isDeleted(reference))
						cover = false; // Archi deleted element
				}				
			}
			
			return cover;
		}
	}
	
	private class ArchiMergePolicy extends DefaultMergePolicy {
		
		@Override public boolean isMandatoryForAddition(EReference reference) {
			return super.isMandatoryForAddition(reference) || 
				reference == IArchimatePackage.eINSTANCE.getRelationship_Source() ||
				reference == IArchimatePackage.eINSTANCE.getRelationship_Target() ||
				reference == IArchimatePackage.eINSTANCE.getDiagramModelConnection_Source() ||
				reference == IArchimatePackage.eINSTANCE.getDiagramModelConnection_Target() ||
				reference == IArchimatePackage.eINSTANCE.getDiagramModelReference_ReferencedModel() ||
				reference == IArchimatePackage.eINSTANCE.getDiagramModelArchimateConnection_Relationship() ||
				reference == IArchimatePackage.eINSTANCE.getDiagramModelArchimateObject_ArchimateElement();
		};
		
		@Override public boolean isMandatoryForDeletion(EReference reference) {
			return super.isMandatoryForDeletion(reference);
		};
		
		@Override
		public Set<EObject> getDeletionGroup(EObject element, IFeaturedModelScope scope) {
			Set<EObject> result = super.getDeletionGroup(element, scope);
			if(localDependentsMap != null) {
				Set<EObject> dependentsSet = localDependentsMap.get(element);
				if(dependentsSet != null) {
					for(EObject dependent : dependentsSet) {
						result.add(dependent);
						result.addAll(getDeletionGroup(dependent, scope));
					}
				}
			}
			return result;
		}
	}
	
	private class ArchiLabelProvider extends DiffMergeLabelProvider {

		@Override
		public String getText(Object element) {
			if(element instanceof EObject)
				return ModelUtils.getLabel((EObject)element);
			else
				return super.getText(element);
		}
	}
	
	private class ArchiScopeDefinition extends AbstractScopeDefinition {

		public ArchiScopeDefinition(IArchimateModel local, String label, boolean editable) {
			super(local, label, editable);
		}
		  
		public IEditableModelScope createScope(Object context_p) {
			return new ArchiModelScope((IArchimateModel)getEntrypoint());
		}
	}
	
	private class ArchiModelScope extends SubtreeModelScope {
		
		public ArchiModelScope (IArchimateModel model) {
			super(model);
		}
		
		@Override
		public boolean add(EObject source, EReference reference, EObject value) {
			return super.add(source, reference, value);
		}
		
		@Override
		public boolean remove(EObject source, EReference reference,	EObject value) {
			if(source instanceof IDiagramModelConnection && (
				reference == IArchimatePackage.eINSTANCE.getDiagramModelConnection_Source() ||
				reference == IArchimatePackage.eINSTANCE.getDiagramModelConnection_Target())) {
				((IDiagramModelConnection)source).disconnect();
				return true;
			}
			else if(source instanceof IDiagramModelArchimateConnection &&
					reference == IArchimatePackage.eINSTANCE.getDiagramModelArchimateConnection_Relationship()) {
				return true;
			}
			else if(source instanceof IDiagramModelArchimateObject &&
					reference == IArchimatePackage.eINSTANCE.getDiagramModelArchimateObject_ArchimateElement()) {
				return true;
			}
			else
				return super.remove(source, reference, value);
		}
	} 
	
	private class ArchiEMFDiffMergeEditorInput extends EMFDiffMergeEditorInput {

		private boolean okPressed;
		
		public ArchiEMFDiffMergeEditorInput(IComparisonMethod method_p) {
			super(method_p);
		}
		
		@Override
		public boolean okPressed() {
			okPressed = super.okPressed();
			return okPressed;
		}
		
		public boolean isOkPressed() {
			return okPressed;
		}		
	}
}
