package it.unibo.cmdb.archimate;

import java.io.StringReader;
import java.text.Normalizer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.datatype.XMLGregorianCalendar;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CommandStack;
import org.eclipse.m2m.atl.emftvm.trace.SourceElement;
import org.eclipse.m2m.atl.emftvm.trace.TargetElement;
import org.eclipse.m2m.atl.emftvm.trace.TraceLink;
import org.eclipse.m2m.atl.emftvm.trace.TraceLinkSet;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSSerializer;
import org.xml.sax.InputSource;

import com.archimatetool.model.IArchimateModel;
import com.archimatetool.model.IArchimatePackage;
import com.archimatetool.model.IIdentifier;
import com.archimatetool.model.INameable;
import com.archimatetool.model.IProperties;
import com.archimatetool.model.IProperty;
import com.archimatetool.model.IRelationship;

import it.unibo.cmdb.archimate.CMDBConfiguration.IdRule;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.CmdbfFactory;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ItemType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.MdrScopedIdType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ModelElement;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ModelRecordType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.RecordType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.RelationshipType;

public class ModelUtils {
		
	public static void makeModelDirty(IArchimateModel model) {
		CommandStack stack = (CommandStack)model.getAdapter(CommandStack.class);
        if(stack != null)
        	stack.execute(new Command(){});
	}
	
	public static String getLabel(EObject eObject) {
		String type = eObject.eClass().getName();
		if(eObject instanceof IRelationship)
			return String.format("<%s>%s(%s -> %s)", type, ((IRelationship)eObject).getName(),
				((IRelationship)eObject).getSource().getName(), ((IRelationship)eObject).getTarget().getName());
		else if(eObject instanceof INameable)
			return String.format("<%s>%s", type, ((INameable)eObject).getName());
		else if(eObject instanceof IProperty)
			return String.format("<%s>%s=%s", type, ((IProperty)eObject).getKey(), ((IProperty)eObject).getValue());				
		else
			return String.format("<%s>", type);
	}
            
    public static void dedupCMDBRelationships(Collection<RelationshipType> relationships) {
		Map<ModelRecordType, Map<ItemType, Map<ItemType, RelationshipType>>> index = new HashMap<ModelRecordType, Map<ItemType,Map<ItemType,RelationshipType>>>();
    	List<RelationshipType> duplicated = new ArrayList<RelationshipType>();
    	
		for(RelationshipType relationship : relationships){
			for(RecordType record : relationship.getRecord()) {
				Map<ItemType, Map<ItemType, RelationshipType>> sourceMap = index.get(record.getType());
				if(sourceMap == null) {
					sourceMap = new HashMap<ItemType, Map<ItemType,RelationshipType>>();
					index.put(record.getType(), sourceMap);
				}
				Map<ItemType, RelationshipType> targetMap = sourceMap.get(relationship.getSourceItem());
				if(targetMap == null) {
					targetMap = new HashMap<ItemType, RelationshipType>();
					sourceMap.put(relationship.getSourceItem(), targetMap);
				}
				RelationshipType oldRelationship = targetMap.get(relationship.getTargetItem());
				if(oldRelationship != null) {
					if(oldRelationship.findInstanceId().compareTo(relationship.findInstanceId()) > 0) {
						duplicated.add(oldRelationship);
						targetMap.put(relationship.getTargetItem(), relationship);
					}
					else
						duplicated.add(relationship);														
				}
				else
					targetMap.put(relationship.getTargetItem(), relationship);
			}
		}
		for(RelationshipType relationship : duplicated)
    		relationships.remove(relationship);
    }    
    
    public static String normalizeFilename(String filename) {
    	String normalized = Normalizer.normalize(filename, Normalizer.Form.NFD);
    	normalized = normalized.replaceAll("\\p{InCombiningDiacriticalMarks}+", "");
    	normalized = normalized.replaceAll("[^a-zA-Z0-9.-_]", "_");
    	int pos = normalized.lastIndexOf('.');
    	if(pos > 0) {
    		String name = normalized.substring(0, pos);
    		String ext = normalized.substring(pos+1);
    		normalized = name.replace('.', '_') + '.' + ext;    		
    	}    	
    	normalized = normalized.replaceAll("_+", "_");
    	return normalized;
    }
	
	public static String getArchiMdrId(IIdentifier element) {
		String id = null;
		List<IdRule> idRules = CMDBPlugin.INSTANCE.getConfiguration().getIdRules();
		if(idRules != null) {
			Iterator<IdRule> iterator = idRules.iterator();
			while(id == null && iterator.hasNext()) {
				IdRule rule = iterator.next();
				EClass ruleType = null;
				if(rule.getType() != null)
					ruleType = (EClass)IArchimatePackage.eINSTANCE.getEClassifier(rule.getType());
				if(ruleType == null || ruleType.isSuperTypeOf(element.eClass()))
					id = rule.mdrId;
			}
		}
		return id;
	}
	
	public static String getArchiLocalId(IIdentifier element) throws Exception {
		String id = null;
		List<IdRule> idRules = CMDBPlugin.INSTANCE.getConfiguration().getIdRules();
		if(idRules != null) {
			Iterator<IdRule> iterator = idRules.iterator();
			while(id == null && iterator.hasNext()) {
				IdRule rule = iterator.next();
				EClass ruleType = null;
				if(rule.getType() != null)
					ruleType = (EClass)IArchimatePackage.eINSTANCE.getEClassifier(rule.getType());
				if(ruleType == null || ruleType.isSuperTypeOf(element.eClass())) {
					String format =  rule.localId;
					final Pattern pattern = Pattern.compile("\\{([^}]+)\\}");
					final Matcher matcher = pattern.matcher(format);
					final StringBuilder builder = new StringBuilder();
					int i = 0;
					while (matcher.find()) {
						String name = matcher.group(1);
						String replacement = null;
						if(name.equals("Id"))
							replacement = element.getId();
						if(name.equals("Type"))
							replacement = element.eClass().getName();
						else if(name.equals("Name") && element instanceof INameable)
							replacement = ((INameable)element).getName();
						else if(name.equals("Source") && element instanceof IRelationship)
							replacement = getArchiLocalId(((IRelationship)element).getSource());
						else if(name.equals("Target") && element instanceof IRelationship)
							replacement = getArchiLocalId(((IRelationship)element).getTarget());
						else if(element instanceof IProperties) {
							IProperty property = null;
							Iterator<IProperty> propertyIterator = ((IProperties)element).getProperties().iterator();
							while(propertyIterator.hasNext() && property==null) {
								IProperty current = propertyIterator.next();
								if(current.getKey().equals(name))
									property = current;
							}
							if(property != null)
								replacement = property.getValue();
						}
						if(replacement == null)
							throw new Exception("Property " + name + " not found: " + ModelUtils.getLabel(element));
						
						builder.append(format.substring(i, matcher.start()));
						builder.append(replacement);
						i = matcher.end();
					}
					builder.append(format.substring(i, format.length()));		
					id = builder.toString();										
				}
			}
		}
		return id;
	}
	
	public static Map<MdrScopedIdType, IIdentifier> updateCMDBElements(TraceLinkSet trace, boolean ignoreMetadata) throws Exception {
		Map<MdrScopedIdType, IIdentifier> idMap = new HashMap<MdrScopedIdType, IIdentifier>();
		for(SourceElement sourceElement : trace.getDefaultSourceElements()) {
			TraceLink traceLink = (TraceLink)sourceElement.eContainer();
			EList<TargetElement>targetList = traceLink.getTargetElements();
			if(sourceElement.getObject() instanceof IIdentifier) {
				IIdentifier source = (IIdentifier)sourceElement.getObject();
				ModelElement target = null;
				Iterator<TargetElement> targetIterator = targetList.iterator();
				while(targetIterator.hasNext() && target==null) {
					TargetElement current = targetIterator.next();
					if(current.getObject() instanceof ModelElement)
						target = (ModelElement) current.getObject();
				}
				if(target != null) {
					String archiMdrId = getArchiMdrId(source);
					String archiLocalId = getArchiLocalId(source);
					if(archiMdrId!=null && archiLocalId!=null) {						
						MdrScopedIdType id = target.addInstanceId(archiMdrId, archiLocalId);
						IIdentifier old = idMap.put(id, source);
						if(old!=null && !old.equals(source))
							throw new Exception("Duplicated id: {" + id + "} " + getLabel(source));
					}
					
					if(!ignoreMetadata) {
						CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(source);					
						String cmdbMdrId = adapter.getMdrId(source);
						String cmdbLocalId = adapter.getLocalId(source);
						if(cmdbMdrId!=null && cmdbLocalId!=null) {
							MdrScopedIdType id = target.addInstanceId(cmdbMdrId, cmdbLocalId);
							IIdentifier old = idMap.put(id, source);
							if(old!=null && !old.equals(source))
								throw new Exception("Duplicated id: {" + id + "} " + getLabel(source));						
						}
						XMLGregorianCalendar lastModified = adapter.getLastModified(source);
						if(lastModified != null) {
							for(RecordType record : target.getRecord())
								record.setLastModified(lastModified);
						}
					}
				}
			}			
		}
		return idMap;		
	}
	
	public static Map<MdrScopedIdType, IIdentifier> updateArchiElements(TraceLinkSet trace) throws Exception {
		Map<MdrScopedIdType, IIdentifier> idMap = new HashMap<MdrScopedIdType, IIdentifier>();
		for(SourceElement sourceElement : trace.getDefaultSourceElements()) {
			TraceLink traceLink = (TraceLink)sourceElement.eContainer();
			EList<TargetElement>targetList = traceLink.getTargetElements();
			if(!targetList.isEmpty()) {
				if(sourceElement.getObject() instanceof ModelElement) {
					ModelElement source = (ModelElement)sourceElement.getObject();
					IIdentifier target = null;
					Iterator<TargetElement> targetIterator = targetList.iterator();
					while(targetIterator.hasNext() && target==null) {
						TargetElement current = targetIterator.next();
						if(current.getObject() instanceof IIdentifier)
							target = (IIdentifier) current.getObject();
					}
					if(target != null) {
						String archiMdrId = getArchiMdrId(target);
						String archiLocalId = getArchiLocalId(target);
						if(archiMdrId!=null && archiLocalId!=null) {						
							MdrScopedIdType id = CmdbfFactory.eINSTANCE.createMdrScopedIdType();
							id.setMdrId(archiMdrId);
							id.setLocalId(archiLocalId);
							IIdentifier old = idMap.put(id, target);
							if(old!=null && !old.equals(target))
								throw new Exception("Duplicated id: {" + id + "} " + getLabel(target));						
						}
						
						CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(target);					
						MdrScopedIdType cmdbId = source.findInstanceId();
						if(cmdbId != null) {
							adapter.setMdrId(target, cmdbId.getMdrId());
							adapter.setLocalId(target, cmdbId.getLocalId());
							IIdentifier old = idMap.put(cmdbId, target);
							if(old!=null && !old.equals(target))
								throw new Exception("Duplicated id: {" + cmdbId + "} " + getLabel(target));						
	
						}
						XMLGregorianCalendar lastModified = null;
						for(RecordType record : source.getRecord()) {
							XMLGregorianCalendar recordLastModified = record.getLastModified();
							if(lastModified==null || lastModified.compare(recordLastModified)<0)
								lastModified = recordLastModified;
						}
						if(lastModified != null)
							adapter.setLastModified(target, lastModified);
					}
				}
			}			
		}
		return idMap;		
	}
	
	public static String toString(Element xml) {
		String value =  null;
		if(xml != null) {
			DOMImplementationLS domImplementation = (DOMImplementationLS) xml.getOwnerDocument().getImplementation();
			LSSerializer lsSerializer = domImplementation.createLSSerializer();
			if(lsSerializer.getDomConfig().canSetParameter("xml-declaration", Boolean.FALSE))
				lsSerializer.getDomConfig().setParameter("xml-declaration", Boolean.FALSE);
			if(lsSerializer.getDomConfig().canSetParameter("format-pretty-print", Boolean.TRUE))
				lsSerializer.getDomConfig().setParameter("format-pretty-print", Boolean.TRUE);			
			value = lsSerializer.writeToString(xml);
		}
		return value!=null ? value : "";
	}
	
	public static Element parseXml(String value) {
		Element xml = null;
		if(value != null) {
		    try {			
			    StringReader reader = new StringReader(value);
			    DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
				documentBuilderFactory.setNamespaceAware(true);
				Document doc = documentBuilderFactory.newDocumentBuilder().parse(new InputSource(reader));
				xml = doc.getDocumentElement();	    		
			} catch (Exception e) {
				IStatus status = new Status(IStatus.ERROR, CMDBPlugin.PLUGIN_ID, e.getMessage(), e);
				CMDBPlugin.INSTANCE.getLog().log(status);
			}
		}		
		return xml;
	}
}
