package it.unibo.cmdbuild.rest;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.CookieStore;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.protocol.ClientContext;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.HttpContext;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;

public class CMDBuildClient {

	public class CMDBuildClientException extends Exception {
		private static final long serialVersionUID = 1L;

		public CMDBuildClientException(String message, Throwable throwable) {
			super(message, throwable);
		}
		
		public CMDBuildClientException(String message) {
			super(message);
		}
	}

	private Map<Integer, Domain> domains = new HashMap<Integer, Domain>();
	private Map<String, LookupType> lookupTypes = new HashMap<String, LookupType>();
	private Map<Integer, Table> classes = new HashMap<Integer, Table>();

	private HttpClient httpclient;
	private CookieStore cookieStore;
	private HttpContext httpContext;
	private URI baseUri;
	private Properties configuration = new Properties();

	public CMDBuildClient(URI uri, String username, String password)
			throws IOException, URISyntaxException, CMDBuildClientException {
		httpclient = new DefaultHttpClient();
		cookieStore = new BasicCookieStore();
		httpContext = new BasicHttpContext();
		httpContext.setAttribute(ClientContext.COOKIE_STORE, cookieStore);
		this.baseUri = uri;
		InputStream stream = getClass().getClassLoader().getResourceAsStream(
				"unibo.properties");
		try {
			configuration.load(stream);
		} finally {
			stream.close();
		}
		try {
			if (!login(username, password))
				throw new CMDBuildClientException("Login failed");
			if (!reload())
				throw new CMDBuildClientException("Unable to load data from cmdb");
		}
		catch(JSONException e) {
			throw new CMDBuildClientException("JSON Exception: " + e.getMessage(), e);
		}
	}

	public Properties getConfiguration() {
		return configuration;
	}

	public Domain getDomain(int id) {
		return domains.get(id);
	}

	public Domain getDomainByName(String name) {
		Domain domain = null;
		Iterator<Domain> iterator = domains.values().iterator();
		while (domain == null && iterator.hasNext()) {
			Domain item = iterator.next();
			if (item.getName().equals(name))
				domain = item;
		}
		return domain;
	}

	public Collection<Domain> domains() {
		return domains.values();
	}

	public LookupType getlookupType(String type) {
		return lookupTypes.get(type);
	}

	public Collection<LookupType> lookupTypes() {
		return lookupTypes.values();
	}

	public Table getClass(int id) {
		return classes.get(id);
	}

	public Table getClassByName(String name) {
		Table clazz = null;
		Iterator<Table> iterator = classes.values().iterator();
		while (clazz == null && iterator.hasNext()) {
			Table item = iterator.next();
			if (item.getName().equals(name))
				clazz = item;
		}
		return clazz;
	}

	public Collection<Table> classes() {
		return classes.values();
	}

	public boolean reload() throws ClientProtocolException, IOException,
			URISyntaxException, JSONException {
		boolean result = true;
		domains.clear();
		classes.clear();
		lookupTypes.clear();
		result &= loadLookupTypes();
		result &= loadLookupValues();
		result &= loadClasses();
		result &= loadClassesAttributes();
		result &= loadDomains();
		result &= loadCards();
		result &= loadRelations();
		return result;
	}

	public void shutdown() {
		httpclient.getConnectionManager().shutdown();
	}

	private JSONObject invoke(String service, String method,
			List<NameValuePair> params) throws ClientProtocolException,
			IOException, URISyntaxException, JSONException {
		JSONObject json = null;
		URI uri = URIUtils
				.createURI(
						baseUri.getScheme(),
						baseUri.getHost(),
						baseUri.getPort(),
						(baseUri.getPath() + "/services/json/" + service + "/" + method)
								.toLowerCase(),
						params != null ? URLEncodedUtils
								.format(params, "UTF-8") : null, null);
		HttpGet httpget = new HttpGet(uri);
		HttpResponse response = httpclient.execute(httpget, httpContext);
		HttpEntity entity = response.getEntity();
		if (entity != null) {
			InputStream stream = entity.getContent();
			try {
				InputStreamReader reader = new InputStreamReader(stream);
				JSONTokener tokener = new JSONTokener(reader);
				json = new JSONObject(tokener);
			} finally {
				stream.close();
			}
		}
		return json;
	}

	private boolean login(String username, String password)
			throws ClientProtocolException, IOException, URISyntaxException,
			JSONException {
		List<NameValuePair> params = new ArrayList<NameValuePair>();
		params.add(new BasicNameValuePair("username", username));
		params.add(new BasicNameValuePair("password", password));
		JSONObject json = invoke("Login", "login", params);
		return json.getBoolean("success");
	}

	private boolean loadLookupTypes() throws ClientProtocolException,
			IOException, URISyntaxException, JSONException {
		JSONObject json = invoke("schema/ModLookup", "getLookupTypeList", null);
		boolean success = json.getBoolean("success");
		if (success) {
			JSONArray jsArray = json.getJSONArray("rows");
			for (int i = 0; i < jsArray.length(); i++) {
				JSONObject jsItem = jsArray.getJSONObject(i);
				String lookupName = jsItem.getString("type");
				List<NameValuePair> params = new ArrayList<NameValuePair>();
				params.add(new BasicNameValuePair("type", lookupName));
				JSONObject jsResult = invoke("schema/ModLookup",
						"getLookupType", params);
				success &= jsResult.getBoolean("success");
				if (success) {
					JSONObject jsLookup = jsResult.getJSONObject("data");
					LookupType lookupType = new LookupType(jsLookup, this);
					assert (!lookupTypes.containsKey(lookupType.getType()));
					lookupTypes.put(lookupType.getType(), lookupType);
				}
			}
		}
		return success;
	}

	private boolean loadLookupValues() throws ClientProtocolException,
			IOException, URISyntaxException, JSONException {
		boolean success = true;
		for (LookupType lookupType : lookupTypes()) {
			List<NameValuePair> params = new ArrayList<NameValuePair>();
			params.add(new BasicNameValuePair("active", "true"));
			params.add(new BasicNameValuePair("type", lookupType.getType()));
			params.add(new BasicNameValuePair("start", "0"));
			params.add(new BasicNameValuePair("limit", "-1"));
			params.add(new BasicNameValuePair("short", "false"));
			JSONObject json = invoke("schema/ModLookup", "getLookupList",
					params);
			success &= json.getBoolean("success");
			if (success) {
				JSONArray jsArray = json.getJSONArray("rows");
				lookupType.parseInstances(jsArray);
			}
		}
		return success;
	}

	private boolean loadClasses() throws ClientProtocolException, IOException,
			URISyntaxException, JSONException {
		List<NameValuePair> params = new ArrayList<NameValuePair>();
		params.add(new BasicNameValuePair("active", "true"));
		JSONObject json = invoke("schema/ModClass", "getAllClasses", params);
		boolean success = json.getBoolean("success");
		if (success) {
			JSONArray jsArray = json.getJSONArray("classes");
			for (int i = 0; i < jsArray.length(); i++) {
				JSONObject jsItem = jsArray.getJSONObject(i);
				Table clazz = new Table(jsItem, this);
				assert (!classes.containsKey(clazz.getId()));
				classes.put(clazz.getId(), clazz);
			}
		}
		return success;
	}

	private boolean loadClassesAttributes() throws ClientProtocolException,
			IOException, URISyntaxException, JSONException {
		boolean success = true;
		for (Table clazz : classes()) {
			List<NameValuePair> params = new ArrayList<NameValuePair>();
			params.add(new BasicNameValuePair("active", "true"));
			params.add(new BasicNameValuePair("idClass", Integer.toString(clazz
					.getId())));
			JSONObject json = invoke("schema/ModClass", "getAttributeList",
					params);
			success &= json.getBoolean("success");
			if (success) {
				JSONArray jsArray = json.getJSONArray("rows");
				clazz.parseAttributes(jsArray);
			}
		}
		return success;
	}

	private boolean loadDomains() throws ClientProtocolException, IOException,
			URISyntaxException, JSONException {
		List<NameValuePair> params = new ArrayList<NameValuePair>();
		params.add(new BasicNameValuePair("active", "true"));
		JSONObject json = invoke("schema/ModClass", "getAllDomains", params);
		boolean success = json.getBoolean("success");
		if (success) {
			JSONArray jsArray = json.getJSONArray("domains");
			for (int i = 0; i < jsArray.length(); i++) {
				JSONObject jsItem = jsArray.getJSONObject(i);
				Domain domain = new Domain(jsItem, this);
				assert (!domains.containsKey(domain.getId()));
				domains.put(domain.getId(), domain);
			}
		}
		return success;
	}

	private boolean loadCards() throws ClientProtocolException, IOException,
			URISyntaxException, JSONException {
		boolean success = true;
		for (Table clazz : classes()) {
			if (!clazz.isSuperClass() && clazz.isActive()) {
				List<NameValuePair> params = new ArrayList<NameValuePair>();
				params.add(new BasicNameValuePair("IdClass", Integer
						.toString(clazz.getId())));
				params.add(new BasicNameValuePair("start", "0"));
				params.add(new BasicNameValuePair("limit", "-1"));
				JSONObject json = invoke("management/ModCard", "getCardList",
						params);
				success &= json.getBoolean("success");
				if (success) {
					JSONArray jsArray = json.getJSONArray("rows");
					clazz.parseInstances(jsArray);
				}
			}
		}
		return success;
	}

	private boolean loadRelations() throws ClientProtocolException,
			IOException, URISyntaxException, JSONException {
		boolean success = true;
		for (Table clazz : classes()) {
			if (!clazz.isSuperClass() && clazz.isActive()) {
				for (Card card : clazz.cards()) {
					List<NameValuePair> params = new ArrayList<NameValuePair>();
					params.add(new BasicNameValuePair("Id", Integer
							.toString(card.getId())));
					params.add(new BasicNameValuePair("IdClass", Integer
							.toString(clazz.getId())));
					params.add(new BasicNameValuePair("domainlimit", "-1"));
					JSONObject json = invoke("management/ModCard",
							"getRelationList", params);
					success &= json.getBoolean("success");
					if (success) {
						JSONArray jsArray = json.getJSONArray("domains");
						for (int i = 0; i < jsArray.length(); i++) {
							JSONObject jsItem = jsArray.getJSONObject(i);
							if (jsItem.getString("src").equals("_1")) {
								Domain domain = getDomain(jsItem.getInt("id"));
								JSONArray jsRelations = jsItem
										.getJSONArray("relations");
								assert (clazz.isSubclassOf(domain.getClass1()));
								for (int j = 0; j < jsRelations.length(); j++) {
									JSONObject jsRelation = jsRelations
											.getJSONObject(j);
									Relation relation = new Relation(card,
											jsRelation, domain);
									relation.getCard1().addRelation(relation,
											Relation.Direction.DIRECT);
									relation.getCard2().addRelation(relation,
											Relation.Direction.INVERSE);
								}
							}
						}
					}
				}
			}
		}
		return success;
	}
}
