package it.unibo.cmdb.cmdbf.client;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.URL;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.namespace.QName;
import javax.xml.transform.stream.StreamSource;
import javax.xml.ws.Dispatch;
import javax.xml.ws.Service;
import javax.xml.ws.soap.SOAPFaultException;

import org.apache.cxf.Bus;
import org.apache.cxf.common.jaxb.JAXBContextCache;
import org.apache.cxf.common.jaxb.JAXBContextCache.CachedContextAndSchemas;
import org.apache.cxf.endpoint.Client;
import org.apache.cxf.endpoint.Endpoint;
import org.apache.cxf.endpoint.EndpointImpl;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.interceptor.Interceptor;
import org.apache.cxf.message.Message;
import org.apache.cxf.phase.AbstractPhaseInterceptor;
import org.apache.cxf.phase.Phase;
import org.apache.cxf.transport.http.HTTPConduit;
import org.apache.cxf.transports.http.configuration.HTTPClientPolicy;
import org.apache.cxf.ws.policy.AbstractPolicyInterceptorProvider;
import org.apache.cxf.ws.policy.AssertionBuilderRegistry;
import org.apache.cxf.ws.policy.AssertionInfo;
import org.apache.cxf.ws.policy.AssertionInfoMap;
import org.apache.cxf.ws.policy.PolicyInterceptorProviderRegistry;
import org.apache.cxf.ws.policy.builder.jaxb.JaxbAssertion;
import org.apache.cxf.ws.security.wss4j.WSS4JOutInterceptor;
import org.apache.neethi.Assertion;
import org.apache.neethi.AssertionBuilderFactory;
import org.apache.neethi.builders.AssertionBuilder;
import org.apache.wss4j.common.ext.WSPasswordCallback;
import org.apache.wss4j.dom.WSConstants;
import org.apache.wss4j.dom.handler.WSHandlerConstants;
import org.dmtf.schemas.cmdbf._1.tns.servicemetadata.QueryServiceMetadata;
import org.dmtf.schemas.cmdbf._1.tns.servicemetadata.RecordTypeList;
import org.dmtf.schemas.cmdbf._1.tns.servicemetadata.RegistrationServiceMetadata;
import org.dmtf.schemas.cmdbf._1.tns.servicemetadata.ServiceDescription;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.BasicExtendedMetaData;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.ExtendedMetaData;
import org.eclipse.emf.ecore.xml.type.AnyType;
import org.eclipse.emf.ecore.xml.type.XMLTypeFactory;
import org.w3c.dom.Element;

import it.unibo.cmdb.cmdbf.client.model.cmdbf.CmdbfFactory;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.CmdbfPackage;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.DeregisterRequestType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.DeregisterResponseType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.DocumentRoot;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.InstanceIdConstraintType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ItemTemplateType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ItemType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.MdrScopedIdType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.Model;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ModelElement;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ModelRecordType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ModelUpdateResponse;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.QueryType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.RecordMetadataType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.RecordType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.RegisterRequestType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.RegisterResponseType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.RelationshipRefType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.RelationshipTemplateType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.RelationshipType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.util.CmdbfResourceFactoryImpl;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.util.CmdbfXMLProcessor;

public class CMDBClient {
	
	private final long timeout = 3600000L;
	private Dispatch<StreamSource> queryDispatch;
	private Dispatch<StreamSource> registrationDispatch;
	private String mdrId;
	private Resource.Factory resourceFactory;
	private CmdbfXMLProcessor xmlProcessor;
	private boolean invalid;
		
	public CMDBClient(URL queryServiceWSDL, QName queryServiceName, QName queryPortName, URL registrationServiceWSDL, QName registrationServiceName, QName registrationPortName, String username, String password) throws Exception {
		System.setProperty("org.apache.cxf.stax.maxChildElements", "1000000");
		
		resourceFactory = new CmdbfResourceFactoryImpl();
		xmlProcessor = new CmdbfXMLProcessor();
		
		Service queryService = Service.create(queryServiceWSDL, queryServiceName);
		queryDispatch = queryService.createDispatch(queryPortName, StreamSource.class, Service.Mode.PAYLOAD);
	    Client queryClient = ((org.apache.cxf.jaxws.DispatchImpl<?>)queryDispatch).getClient();
        Endpoint queryEndpoint = queryClient.getEndpoint();
	    setupPolicyInterceptor(queryEndpoint);
        setupAuthentication(queryEndpoint, username, password);
        setupTimeout(queryClient, timeout);
        
        Service registrationService = Service.create(registrationServiceWSDL, registrationServiceName);
        registrationDispatch = registrationService.createDispatch(registrationPortName, StreamSource.class, Service.Mode.PAYLOAD);
	    Client registrationClient = ((org.apache.cxf.jaxws.DispatchImpl<?>)registrationDispatch).getClient();
        Endpoint registrationEndpoint = registrationClient.getEndpoint();
	    setupPolicyInterceptor(registrationEndpoint);
        setupAuthentication(registrationEndpoint, username, password);
        setupTimeout(registrationClient, timeout);                        
	}
	
	public boolean isInvalid() {
		return invalid;
	}
	
	public QueryType newQuery() {
		QueryType query = CmdbfFactory.eINSTANCE.createQueryType();
		query.setMdrId(mdrId);
		query.setModelMetadata(xmlProcessor.getModelMetadata());
		return query;
	}
		
	public QueryType newQuery(String query) throws Exception {
		QueryType cmdbfQuery = null;
		if(query != null){
			InputStream stream = new ByteArrayInputStream(query.getBytes("UTF-8"));
	        Resource resource = xmlProcessor.load(stream, null);
			DocumentRoot documentRoot = (DocumentRoot)resource.getContents().get(0);
			cmdbfQuery = documentRoot.getQuery();
			cmdbfQuery.setModelMetadata(xmlProcessor.getModelMetadata());
			resource.getContents().clear();
		}
		return cmdbfQuery;
	}
	
	public Model newModel() {
		Model model = CmdbfFactory.eINSTANCE.createModel();
		model.setMdrId(mdrId);
		model.setModelMetadata(xmlProcessor.getModelMetadata());
		return model;
	}
	
	public Model query(String query) throws Exception {		
		return query(newQuery(query));
	}
	
	public Model query(QueryType query) throws Exception {
		DocumentRoot root = CmdbfFactory.eINSTANCE.createDocumentRoot();
		root.setQuery(query);
		Model model = invoke(queryDispatch, root).getModel();
		root.setQuery(null);
		
		if(model != null)
			model.setMdrId(mdrId);		
    	return model;
	}
	
	public ModelUpdateResponse update(Model model) throws Exception {
		ModelUpdateResponse response = CmdbfFactory.eINSTANCE.createModelUpdateResponse();
		
		QueryType deregisterQuery = null;;
		if(!model.getRemovedItemId().isEmpty()) {
			deregisterQuery = CmdbfFactory.eINSTANCE.createQueryType();
			String TEMPLATE_PREFIX = "__remove_";
			ItemTemplateType sourceItemTemplate = CmdbfFactory.eINSTANCE.createItemTemplateType();
			sourceItemTemplate.setId(TEMPLATE_PREFIX + "SourceItems");
			InstanceIdConstraintType idConstraint = CmdbfFactory.eINSTANCE.createInstanceIdConstraintType();
			sourceItemTemplate.setInstanceIdConstraint(idConstraint);
			idConstraint.getInstanceId().addAll(EcoreUtil.copyAll(model.getRemovedItemId()));
			sourceItemTemplate.setSuppressFromResult(true);
			deregisterQuery.getItemTemplate().add(sourceItemTemplate);
			
			ItemTemplateType targetItemTemplate = EcoreUtil.copy(sourceItemTemplate);
			targetItemTemplate.setId(TEMPLATE_PREFIX + "TargetItems");
			deregisterQuery.getItemTemplate().add(targetItemTemplate);
			
			RelationshipTemplateType relationshipSourceTemplate = CmdbfFactory.eINSTANCE.createRelationshipTemplateType();
			relationshipSourceTemplate.setId(TEMPLATE_PREFIX + "RelationshipsBySource");
			RelationshipRefType relationshipSourceRef = CmdbfFactory.eINSTANCE.createRelationshipRefType();
			relationshipSourceRef.setRef(sourceItemTemplate.getId());
			relationshipSourceTemplate.setSourceTemplate(relationshipSourceRef);
			deregisterQuery.getRelationshipTemplate().add(relationshipSourceTemplate);
			
			RelationshipTemplateType relationshipTargetTemplate = CmdbfFactory.eINSTANCE.createRelationshipTemplateType();
			relationshipTargetTemplate.setId(TEMPLATE_PREFIX + "RelationshipsByTarget");
			RelationshipRefType relationshipTergetRef = CmdbfFactory.eINSTANCE.createRelationshipRefType();
			relationshipTergetRef.setRef(targetItemTemplate.getId());
			relationshipTargetTemplate.setTargetTemplate(relationshipTergetRef);
			deregisterQuery.getRelationshipTemplate().add(relationshipTargetTemplate);
		}
		
		if(deregisterQuery != null) {
			DocumentRoot queryRoot = CmdbfFactory.eINSTANCE.createDocumentRoot();
			queryRoot.setQuery(deregisterQuery);
			Model result = invoke(queryDispatch, queryRoot).getModel();
			for(RelationshipType relationship : result.getRelationship())
				model.getRemovedRelationshipId().add(EcoreUtil.copy(relationship.findInstanceId()));
		}
		
		if(!model.getRemovedItemId().isEmpty() || !model.getRemovedRelationshipId().isEmpty()) {
			DeregisterRequestType deregister = model.getDeregister();			
			DocumentRoot deregisterRoot = CmdbfFactory.eINSTANCE.createDocumentRoot();
			deregisterRoot.setDeregisterRequest(deregister);
			DeregisterResponseType deregisterResponse = invoke(registrationDispatch, deregisterRoot).getDeregisterResponse();
			response.getInstanceResponse().addAll(deregisterResponse.getDeregisterInstanceResponse());
			model.setDeregister(deregister);
			for(MdrScopedIdType id : model.getRemovedRelationshipId()) {
				RelationshipType relationship = model.findRelationship(id);
				if(relationship != null)
					model.getRelationship().remove(relationship);
			}
			model.getRemovedRelationshipId().clear();
			for(MdrScopedIdType id : model.getRemovedItemId()) {
				ItemType item = model.findItem(id);
				if(item != null)
					model.getItem().remove(item);
			}
			model.getRemovedItemId().clear();
		}
		
		if(!model.getItem().isEmpty() || !model.getRelationship().isEmpty()) {
			RegisterRequestType register = model.getRegister();
			DocumentRoot registerRoot = CmdbfFactory.eINSTANCE.createDocumentRoot();
			registerRoot.setRegisterRequest(register);
			RegisterResponseType registerResponse = invoke(registrationDispatch, registerRoot).getRegisterResponse();
			response.getInstanceResponse().addAll(registerResponse.getRegisterInstanceResponse());
			model.setRegister(register);
		}
		return response;
	}
	
	public Model load(InputStream stream) throws IOException {
		Model model = null;
		Resource resource = xmlProcessor.load(stream, null);
		if(!resource.getContents().isEmpty()) {
			EObject content = resource.getContents().get(0);
			if(content instanceof DocumentRoot) {
				DocumentRoot documentRoot = (DocumentRoot)content;
				model = documentRoot.getModel();
				documentRoot.getMixed().clear();
			}
		}
		resource.getContents().clear();
		return model;
	}
	
	public void save(Model model, OutputStream stream) throws IOException {
		Resource resource = resourceFactory.createResource(URI.createFileURI(UUID.randomUUID().toString()));
		DocumentRoot root = CmdbfFactory.eINSTANCE.createDocumentRoot();
	    resource.getContents().add(root);
	    root.setModel(model);	
		xmlProcessor.save(stream, resource, null);
		root.getMixed().clear();
		resource.getContents().clear();
	}
	
	public Model diff(Model model, Model baseModel) throws Exception {
		Model diffModel = newModel();
		
		if(baseModel.getMdrId() != null)
			diffModel.setMdrId(baseModel.getMdrId());						
		
		for(ItemType item : model.getItem()) {
			ItemType diffItem = null;
			ItemType baseItem = baseModel.findItem(item); 
			if(baseItem != null)
				diffItem = diff(item, baseItem);
			else
				diffItem = EcoreUtil.copy(item);
			if(diffItem != null)
				diffModel.getItem().add(diffItem);
		}
				
		for(MdrScopedIdType removed : model.getRemovedItemId()) {
			ItemType baseItem = baseModel.findItem(removed);
			if(baseItem != null) {
				ItemType diffItem = diffModel.findItem(baseItem);
				if(diffItem == null) {
					diffItem = EcoreUtil.copy(baseItem);
					diffModel.getItem().add(diffItem);
				}
				diffModel.remove(diffItem);
			}
		}
				
		for(ItemType baseItem : baseModel.getItem()) {
			if(!baseItem.getRecord().isEmpty()) {
				ItemType item = model.findItem(baseItem); 
				if(item == null) {
					ItemType diffItem = diffModel.findItem(baseItem);
					if(diffItem == null) {
						diffItem = EcoreUtil.copy(baseItem);
						diffModel.getItem().add(diffItem);
					}
					diffModel.remove(diffItem);
				}
			}
		}
		
		for(RelationshipType relationship : model.getRelationship()) {
			RelationshipType diffRelationship;
			RelationshipType baseRelationship = baseModel.findRelationship(relationship); 
			if(baseRelationship != null) {
				 diffRelationship = diff(relationship, baseRelationship);
			}
			else
				diffRelationship = EcoreUtil.copy(relationship);
			
			if(diffRelationship != null) {
				if(relationship.getSourceItem() != null) {
					ItemType diffSource = diffModel.findItem(relationship.getSourceItem());
					if(diffSource == null) {
						diffSource = diffModel.addItem();
						for(MdrScopedIdType id : relationship.getSourceItem().getInstanceId())						
							diffSource.getInstanceId().add(EcoreUtil.copy(id));
					}
					diffRelationship.setSourceItem(diffSource);
				}
				if(relationship.getTargetItem() != null) {
					ItemType diffTarget = diffModel.findItem(relationship.getTargetItem());
					if(diffTarget == null) {
						diffTarget = diffModel.addItem();
						for(MdrScopedIdType id : relationship.getTargetItem().getInstanceId())						
							diffTarget.getInstanceId().add(EcoreUtil.copy(id));
					}
					diffRelationship.setTargetItem(diffTarget);
				}
				diffModel.getRelationship().add(diffRelationship);
			}
		}
		
		for(MdrScopedIdType removed : model.getRemovedRelationshipId()) {
			RelationshipType baseRelationship = baseModel.findRelationship(removed);
			if(baseRelationship != null) {
				RelationshipType diffRelationship = diffModel.findRelationship(baseRelationship);				
				if(diffRelationship == null) {				
					diffRelationship = EcoreUtil.copy(baseRelationship);
					if(baseRelationship.getSourceItem() != null) {					
						ItemType diffSource = diffModel.findItem(baseRelationship.getSourceItem());
						if(diffSource != null)
							diffRelationship.setSourceItem(diffSource);
					}
					if(baseRelationship.getTargetItem() != null) {
						ItemType diffTarget = diffModel.findItem(baseRelationship.getTargetItem());
						if(diffTarget != null)
							diffRelationship.setTargetItem(diffTarget);
					}				
					diffModel.getRelationship().add(diffRelationship);
				}
				diffModel.remove(diffRelationship);
			}
		}
		
		for(RelationshipType baseRelationship : baseModel.getRelationship()) {
			RelationshipType relationship = model.findRelationship(baseRelationship); 
			if(relationship == null) {
				RelationshipType diffRelationship = diffModel.findRelationship(baseRelationship);				
				if(diffRelationship == null) {				
					diffRelationship = EcoreUtil.copy(baseRelationship);
					if(baseRelationship.getSourceItem() != null) {					
						ItemType diffSource = diffModel.findItem(baseRelationship.getSourceItem());
						if(diffSource != null)
							diffRelationship.setSourceItem(diffSource);
					}
					if(baseRelationship.getTargetItem() != null) {
						ItemType diffTarget = diffModel.findItem(baseRelationship.getTargetItem());
						if(diffTarget != null)
							diffRelationship.setTargetItem(diffTarget);
					}				
					diffModel.getRelationship().add(diffRelationship);
				}				
				diffModel.remove(diffRelationship);
			}
		}
					
		return diffModel;
	}
	
	@SuppressWarnings("unchecked")
	private <T extends ModelElement> T diff(T element, T baseElement) throws Exception {
		T diffElement = null;
		
		Iterator<RecordType> recordIterator = element.getRecord().iterator();
		while(recordIterator.hasNext()) {
			RecordType record = recordIterator.next();
			RecordType diffRecord = null;
			RecordType baseRecord = null;
			for(RecordType currentRecord : baseElement.getRecord()) {
				if(record.getType() == currentRecord.getType()) {								
					if(record.getRecordId()==null || record.getRecordId().isEmpty() || currentRecord.getRecordId()==null || currentRecord.getRecordId().isEmpty() || record.getRecordId().equals(currentRecord.getRecordId())) {
						if(baseRecord == null)
							baseRecord = currentRecord;
						else
							throw new IllegalArgumentException("Found more than one record: " + record);
					}
				}				
			}
			if(baseRecord != null) {
				diffRecord = diff(record, baseRecord);
			}
			else
				diffRecord = EcoreUtil.copy(record);
			if(diffRecord != null) {
				if(diffElement == null) {
					diffElement = (T)CmdbfFactory.eINSTANCE.create(element.eClass());
					for(MdrScopedIdType id : element.getInstanceId())						
						diffElement.getInstanceId().add(EcoreUtil.copy(id));
					for(MdrScopedIdType id : baseElement.getInstanceId()) {
						boolean found = false;
						for(MdrScopedIdType existingId : diffElement.getInstanceId())
							found |= id.equals(existingId);
						if(!found)
							diffElement.getInstanceId().add(EcoreUtil.copy(id));
					}
				}
				diffElement.getRecord().add(diffRecord);
			}
		}				
		return diffElement;		
	}
		
	private RecordType diff(RecordType record, RecordType baseRecord) throws Exception {
		RecordType diffRecord = null;
		ModelRecordType recordType = record.getType();
		EClass recordEClass = (EClass)xmlProcessor.getModelMetadata().getElement(recordType.getNamespace(), recordType.getLocalName()).getEType();
		
		for(String name : record.getPropertyName()) {
			if(record.isSet(name)) {
				boolean propertyModified = false;
				Object value = record.get(name);
				if(baseRecord.has(name) && baseRecord.isSet(name)) {
					Object baseValue = baseRecord.get(name);					
					if(value instanceof EObject && baseValue instanceof EObject) {
						EcoreUtil.EqualityHelper diffEqualityHelper = new EcoreUtil.EqualityHelper(){
							private static final long serialVersionUID = 1L;						
							@Override
							protected boolean haveEqualFeature(EObject eObject, EObject eBaseObject, EStructuralFeature feature) {
								if (eObject.eIsSet(feature))
									return super.haveEqualFeature(eObject, eBaseObject, feature);
								else
									return true;								
							}
						};  																	
						EObject eValue = (EObject)value;
						EObject eBaseValue = (EObject)baseValue;
						if(!diffEqualityHelper.equals(eValue, eBaseValue)) {
							if(eValue.eClass().equals(eBaseValue.eClass())) {
								EStructuralFeature mdrIdFeature = eValue.eClass().getEStructuralFeature(CmdbfPackage.eINSTANCE.getMdrScopedIdType_MdrId().getName());
								EStructuralFeature localIdFeature = eValue.eClass().getEStructuralFeature(CmdbfPackage.eINSTANCE.getMdrScopedIdType_LocalId().getName());
								if(mdrIdFeature instanceof EAttribute && localIdFeature instanceof EAttribute) {
									String mdrId = (String)eValue.eGet(mdrIdFeature);
									String localId = (String)eValue.eGet(localIdFeature);
									String baseMdrId = (String)eValue.eGet(mdrIdFeature);
									String baseLocalId = (String)eValue.eGet(localIdFeature);
									Model baseModel = baseRecord.getParent().getParent().getParent();
									ItemType item = null;
									if(mdrId != null && localId != null)
										 item = baseModel.findItem(mdrId, localId);
									ItemType baseItem = null;
									if(baseMdrId != null && baseLocalId != null)
										 baseItem = baseModel.findItem(baseMdrId, baseLocalId);
									if(item==null || baseItem==null || !item.equals(baseItem))
										propertyModified = true;													
								}
								else
									propertyModified = true;
							}
							else
								propertyModified = true;
						}										
					}
					else if(value!=null)
						propertyModified = !value.equals(baseValue);
					else
						propertyModified = !(value==null && baseValue==null);																							
				}
				else
					propertyModified = true;
				
				if(propertyModified) {
					if(diffRecord == null) {
						diffRecord = CmdbfFactory.eINSTANCE.createRecordType();						
						EStructuralFeature feature = xmlProcessor.getModelMetadata().getElement(recordType.getNamespace(), recordType.getLocalName());		
						EClass eClass = (EClass)feature.getEType();
						EObject content = eClass.getEPackage().getEFactoryInstance().create(eClass);
						diffRecord.getAny().set(feature, content);
						
						ExtendedMetaData metadata = new BasicExtendedMetaData();
						RecordMetadataType diffRecordMetadata = CmdbfFactory.eINSTANCE.createRecordMetadataType();
						AnyType diffMetadata = XMLTypeFactory.eINSTANCE.createAnyType();
						EStructuralFeature diffFeature = metadata.demandFeature("cmdb", "diff", true);
						diffRecordMetadata.getAny().add(diffFeature, diffMetadata);
						EStructuralFeature modifiedFeature = metadata.demandFeature(null, "modified", false);
						diffMetadata.eSet(modifiedFeature, true);
						diffRecord.setRecordMetadata(diffRecordMetadata);
						
//						PropertySetType propertySet = CmdbfFactory.eINSTANCE.createPropertySetType();
//						propertySet.setNamespace(recordType.getNamespace());
//						propertySet.setLocalName(recordType.getLocalName());
//						diffRecord.setPropertySet(propertySet);
					}
					EStructuralFeature feature = recordEClass.getEStructuralFeature(name);
					Object diffValue = value instanceof EObject ? EcoreUtil.copy((EObject)value) : value;
					diffRecord.getContent().eSet(feature, diffValue);
					//diffRecord.getPropertySet().getAny().add(feature, value);
				}
			}
		}		
		return diffRecord;		
	}	
	
	public void addPrefix(String prefix, String namespace) {
		xmlProcessor.addPrefix(prefix, namespace);
	}
	
	public ModelRecordType getRecordType(String namespace, String localName) {
		return xmlProcessor.getModelMetadata().getRecordType(namespace, localName);
	}

		
	public void setTimeout(long timeout) {
		Client queryClient = ((org.apache.cxf.jaxws.DispatchImpl<?>)queryDispatch).getClient();
        Client registrationClient = ((org.apache.cxf.jaxws.DispatchImpl<?>)registrationDispatch).getClient();
        setupTimeout(queryClient, timeout);
        setupTimeout(registrationClient, timeout);
	}
	
	private void setupTimeout(Client client, long timeout) {
		HTTPConduit conduit = (HTTPConduit) client.getConduit();
        HTTPClientPolicy policy = new HTTPClientPolicy();
        policy.setConnectionTimeout(timeout);
        policy.setReceiveTimeout(timeout);
        conduit.setClient(policy);
	}
    	
	private void setupAuthentication(Endpoint cxfEndpoint, final String username, final String password) {
		Map<String, Object> outProps = new HashMap<String, Object>();
		outProps.put(WSHandlerConstants.ACTION, WSHandlerConstants.USERNAME_TOKEN);
		outProps.put(WSHandlerConstants.PASSWORD_TYPE, WSConstants.PW_DIGEST);
		WSS4JOutInterceptor wssOut = new WSS4JOutInterceptor(outProps);
		cxfEndpoint.getOutInterceptors().add((Interceptor<? extends Message>) wssOut);        
        final CallbackHandler pwdCallback = new CallbackHandler() {

			@Override
			public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException {
				WSPasswordCallback pc = (WSPasswordCallback) callbacks[0];
				if (username.equals(pc.getIdentifier())) {
		            pc.setPassword(password);
		        }
			}
        	
        };
        outProps.put(WSHandlerConstants.USER, username);
        outProps.put(WSHandlerConstants.PW_CALLBACK_REF, pwdCallback);
	}
	
	private void setupPolicyInterceptor(Endpoint cxfEndpoint) {
		Bus bus = ((EndpointImpl)cxfEndpoint).getBus();
		AssertionBuilderRegistry abr = bus.getExtension(AssertionBuilderRegistry.class);
		abr.registerBuilder(new CMDBServiceMetadataAssertionBuilder());
		PolicyInterceptorProviderRegistry pipr = bus.getExtension(PolicyInterceptorProviderRegistry.class);
		pipr.register(new CMDBPolicyInterceptorProvider());
	}
	
	private void setQueryServiceMetadata(QueryServiceMetadata metadata) {
		ServiceDescription serviceDescription = metadata.getServiceDescription();
		if(serviceDescription != null)
			mdrId = serviceDescription.getMdrId();
		RecordTypeList recordTypeList = metadata.getRecordTypeList();
		if(recordTypeList != null) {
			try {
				xmlProcessor.parseRecordTypeList(recordTypeList);
			} catch (Exception e) {
				throw new Error(e);
			}
		}
	}
	
	private void setRegistrationServiceMetadata(RegistrationServiceMetadata metadata) {
		ServiceDescription serviceDescription = metadata.getServiceDescription();
		if(serviceDescription != null)
			mdrId = serviceDescription.getMdrId();
		RecordTypeList recordTypeList = metadata.getRecordTypeList();
		if(recordTypeList != null) {
			try {
				xmlProcessor.parseRecordTypeList(recordTypeList);
			} catch (Exception e) {
				throw new Error(e);
			}
		}
	}
	
	private DocumentRoot invoke(Dispatch<StreamSource> dispatch, EObject eObject) throws Exception {
        final Resource inputResource = resourceFactory.createResource(URI.createFileURI(UUID.randomUUID().toString()));
        inputResource.getContents().add(eObject);
        
        final PipedInputStream in = new PipedInputStream();
        final PipedOutputStream out = new PipedOutputStream(in);
        new Thread(
        	new Runnable(){
        		public void run(){
        	        try {
						xmlProcessor.save(out, inputResource, null);
						out.close();
					} catch (IOException e) {
						throw new Error(e);
					}
        		}
        	}
        ).start();        
        StreamSource inputSource = new StreamSource(in);
        
        StreamSource outputSource = null;
        try {
        	outputSource = dispatch.invoke(inputSource);
        }
        catch(SOAPFaultException e) {
        	invalid = true;
        	throw e;
        }
	
        Resource outputResource = xmlProcessor.load(outputSource.getInputStream(), null);
        DocumentRoot documentRoot = (DocumentRoot)outputResource.getContents().get(0);
        
		inputResource.getContents().clear();
		outputResource.getContents().clear();
        
        if(documentRoot.getDeregistrationErrorFault() != null)
        	throw new DeregistrationException(documentRoot.getDeregistrationErrorFault());
        if(documentRoot.getExpensiveQueryErrorFault() != null)
        	throw new ExpensiveQueryException(documentRoot.getExpensiveQueryErrorFault());
        if(documentRoot.getInvalidMDRFault() != null)
        	throw new InvalidMDRException(documentRoot.getInvalidMDRFault());
        if(documentRoot.getInvalidPropertyTypeFault() != null)
        	throw new InvalidPropertyException(documentRoot.getInvalidPropertyTypeFault());
        if(documentRoot.getInvalidRecordFault() != null)
        	throw new InvalidRecordException(documentRoot.getInvalidRecordFault());
        if(documentRoot.getQueryErrorFault() != null)
        	throw new QueryException(documentRoot.getQueryErrorFault());        
        if(documentRoot.getRegistrationErrorFault() != null)
        	throw new RegistrationException(documentRoot.getRegistrationErrorFault());
        if(documentRoot.getUnknownTemplateIDFault() != null)
        	throw new UnknownTemplateIDException(documentRoot.getUnknownTemplateIDFault());        
        if(documentRoot.getUnsupportedConstraintFault() != null)
        	throw new UnsupportedConstraintException(documentRoot.getUnsupportedConstraintFault());
        if(documentRoot.getUnsupportedRecordTypeFault() != null)
        	throw new UnsupportedRecordTypeException(documentRoot.getUnsupportedRecordTypeFault());
        if(documentRoot.getUnsupportedSelectorFault() != null)
        	throw new UnsupportedSelectorException(documentRoot.getUnsupportedSelectorFault());
        if(documentRoot.getXPathErrorFault() != null)
        	throw new XPathException(documentRoot.getXPathErrorFault());
        else
        	return documentRoot;
	}
	
	private class CMDBPolicyInterceptorProvider extends AbstractPolicyInterceptorProvider{
		private static final long serialVersionUID = 2268274116876723831L;

		public CMDBPolicyInterceptorProvider() {
			super(Arrays.asList(new CMDBServiceMetadataAssertionBuilder().getKnownElements()));
			CMDBAssertionsInterceptor interceptor = new CMDBAssertionsInterceptor(); 
	        getInInterceptors().add(interceptor);
	        getOutInterceptors().add(interceptor);
	        getInFaultInterceptors().add(interceptor);
	        getOutFaultInterceptors().add(interceptor);
		}
		
		private class CMDBAssertionsInterceptor extends AbstractPhaseInterceptor<Message> {
			public CMDBAssertionsInterceptor() {
	            super(Phase.POST_LOGICAL);
	        }

			@Override
			public void handleMessage(Message message) throws Fault {

	            AssertionInfoMap aim = message.get(AssertionInfoMap.class);
	            for (QName an : getAssertionTypes()) {
	                Collection<AssertionInfo> ais = aim.getAssertionInfo(an);
	                if (null != ais) {
	                    for (AssertionInfo ai : ais) {
	                    	ai.setAsserted(true);
	                    }
	                }
	            }
			}
		}
	}
	
	private class CMDBServiceMetadataAssertionBuilder implements AssertionBuilder<Element> {
		private final QName QUERY_QNAME = new QName("http://schemas.dmtf.org/cmdbf/1/tns/serviceMetadata", "queryServiceMetadata");
		private final QName REGISTRATION_QNAME = new QName("http://schemas.dmtf.org/cmdbf/1/tns/serviceMetadata", "registrationServiceMetadata");

		@Override
		public Assertion build(Element element,	AssertionBuilderFactory factory) throws IllegalArgumentException {
	        try {
	        	Assertion assertion = null;
				if(QUERY_QNAME.getNamespaceURI().equals(element.getNamespaceURI()) && 
        			QUERY_QNAME.getLocalPart().equals(element.getLocalName())){
					CachedContextAndSchemas ccs = JAXBContextCache.getCachedContextAndSchemas(QueryServiceMetadata.class);
			    	JAXBContext ctx = ccs.getContext();
			        Unmarshaller unmarshaller = ctx.createUnmarshaller();
			        QueryServiceMetadata metadata = unmarshaller.unmarshal(element, QueryServiceMetadata.class).getValue();
					JaxbAssertion<QueryServiceMetadata> queryAssertion = new JaxbAssertion<QueryServiceMetadata>(QUERY_QNAME, false);
					queryAssertion.setData(metadata);
					assertion = queryAssertion;
					setQueryServiceMetadata(metadata);
	        	}
	        	else if(REGISTRATION_QNAME.getNamespaceURI().equals(element.getNamespaceURI()) && 
        			REGISTRATION_QNAME.getLocalPart().equals(element.getLocalName())){
	        		CachedContextAndSchemas ccs = JAXBContextCache.getCachedContextAndSchemas(RegistrationServiceMetadata.class);
			    	JAXBContext ctx = ccs.getContext();
			        Unmarshaller unmarshaller = ctx.createUnmarshaller();
			        RegistrationServiceMetadata metadata = unmarshaller.unmarshal(element, RegistrationServiceMetadata.class).getValue();
					JaxbAssertion<RegistrationServiceMetadata> registrationAssertion = new JaxbAssertion<RegistrationServiceMetadata>(REGISTRATION_QNAME, false);
					registrationAssertion.setData(metadata);
					assertion = registrationAssertion;
					setRegistrationServiceMetadata(metadata);
	        	}
				return assertion;
			} catch (JAXBException e) {
				throw new Error(e);
			}
		}

		@Override
		public QName[] getKnownElements() {
			return new QName[]{QUERY_QNAME, REGISTRATION_QNAME};
		}
	}

	public String getMdrId() {
		return mdrId;
	}	
}