package org.cmdbuild.cmdbf.client.graph;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.jgrapht.DirectedGraph;
import org.jgrapht.EdgeFactory;
import org.jgrapht.VertexFactory;

import it.unibo.cmdb.cmdbf.client.model.cmdbf.CmdbfFactory;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ItemType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.Model;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.RelationshipType;

public class CmdbfGraph implements DirectedGraph<ItemType, RelationshipType>, EdgeFactory<ItemType, RelationshipType>, VertexFactory<ItemType> {
	
	private Model model;
	private CmdbfFactory factory;
	
	public CmdbfGraph(Model model) {
		this.model = model;
		this.factory = (CmdbfFactory) model.eClass().getEPackage().getEFactoryInstance();
	}

	@Override
	public RelationshipType addEdge(ItemType source, ItemType target) {
		RelationshipType relationship = createEdge(source, target);
		if(!addEdge(source, target, relationship)) {
			relationship.setSourceItem(null);
			relationship.setTargetItem(null);
			relationship = null;
		}
		return relationship;
	}

	@Override
	public boolean addEdge(ItemType source, ItemType target, RelationshipType relationship) {
		return model.getRelationship().add(relationship);
	}

	@Override
	public boolean addVertex(ItemType item) {
		return model.getItem().add(item);
	}

	@Override
	public boolean containsEdge(RelationshipType relationship) {
		return relationship.getParent().getParent() == model;
	}

	@Override
	public boolean containsEdge(ItemType source, ItemType target) {
		return getEdge(source, target) != null;
	}

	@Override
	public boolean containsVertex(ItemType item) {
		return item.getParent().getParent() == model;
	}

	@Override
	public Set<RelationshipType> edgeSet() {
		return new ListSet<RelationshipType>(model.getRelationship());
	}

	@Override
	public Set<RelationshipType> edgesOf(ItemType item) {
		Set<RelationshipType> edges = new HashSet<RelationshipType>();
		edges.addAll(item.getRelationshipBySource());
		edges.addAll(item.getRelationshipByTarget());
		return edges;
	}

	@Override
	public Set<RelationshipType> getAllEdges(ItemType source, ItemType target) {
		Set<RelationshipType> edges = new HashSet<RelationshipType>();
		for(RelationshipType relationship : source.getRelationshipBySource()) {
			if(relationship.getTargetItem() == target)
				edges.add(relationship);
		}
		for(RelationshipType relationship : source.getRelationshipByTarget()) {
			if(relationship.getSourceItem() == target)
				edges.add(relationship);
		}
		return edges;
	}

	@Override
	public RelationshipType getEdge(ItemType source, ItemType target) {
		RelationshipType relationship = null;
		for(RelationshipType element : source.getRelationshipBySource()) {
			if(element.getTargetItem() == target) {
				relationship = element;
				break;
			}
		}
		return relationship;
	}

	@Override
	public EdgeFactory<ItemType, RelationshipType> getEdgeFactory() {
		return this;
	}

	@Override
	public ItemType getEdgeSource(RelationshipType relationship) {
		return relationship.getSourceItem();
	}

	@Override
	public ItemType getEdgeTarget(RelationshipType relationship) {
		return relationship.getTargetItem();
	}

	@Override
	public double getEdgeWeight(RelationshipType relationship) {
		return 0;
	}

	@Override
	public boolean removeAllEdges(Collection<? extends RelationshipType> relationships) {
		boolean removed = true;
		for(RelationshipType relationship : relationships)
			removed &= removeEdge(relationship);
		return removed;
	}

	@Override
	public Set<RelationshipType> removeAllEdges(ItemType source, ItemType target) {
		Set<RelationshipType> removed = new HashSet<RelationshipType>();
		for(RelationshipType relationship : source.getRelationshipBySource()) {
			if(relationship.getTargetItem() == target)
				removed.add(relationship);
		}
		removeAllEdges(removed);
		return removed;
	}

	@Override
	public boolean removeAllVertices(Collection<? extends ItemType> items) {
		boolean removed = true;
		for(ItemType item : items)
			removed &= removeVertex(item);
		return removed;
	}

	@Override
	public boolean removeEdge(RelationshipType relationship) {
		boolean removed = false;
		if(relationship.getParent().getParent() != model) {
			relationship.setSourceItem(null);
			relationship.setTargetItem(null);
			removed = relationship.getParent().getParent().getRelationship().remove(relationship);					
		}
		return removed;
	}

	@Override
	public RelationshipType removeEdge(ItemType source, ItemType target) {
		RelationshipType relationship = getEdge(source, target);
		if(relationship != null)
			removeEdge(relationship);
		return relationship;
	}

	@Override
	public boolean removeVertex(ItemType item) {
		boolean removed = false;
		if(item.getParent().getParent() != model)
			removed = item.getParent().getParent().getItem().remove(item);
		return removed;
	}

	@Override
	public Set<ItemType> vertexSet() {
		return new ListSet<ItemType>(model.getItem());
	}

	@Override
	public int inDegreeOf(ItemType item) {
		return item.getRelationshipByTarget().size();
	}

	@Override
	public Set<RelationshipType> incomingEdgesOf(ItemType item) {
		return new ListSet<RelationshipType>(item.getRelationshipByTarget());
	}

	@Override
	public int outDegreeOf(ItemType item) {
		return item.getRelationshipBySource().size();
	}

	@Override
	public Set<RelationshipType> outgoingEdgesOf(ItemType item) {
		return new ListSet<RelationshipType>(item.getRelationshipBySource());
	}

	@Override
	public RelationshipType createEdge(ItemType source, ItemType target) {
		RelationshipType relationship = factory.createRelationshipType();
		relationship.setSourceItem(source);
		relationship.setTargetItem(target);
		return relationship;
	}

	@Override
	public ItemType createVertex() {
		return factory.createItemType();
	}
	
	private static class ListSet<T> implements Set<T> {
		
		private List<T> list;
		
		public ListSet(List<T> list) {
			this.list = list;
		}

		@Override
		public int size() {
			return list.size();
		}

		@Override
		public boolean isEmpty() {
			return list.isEmpty();
		}

		@Override
		public boolean contains(Object o) {
			return list.contains(o);
		}

		@Override
		public Iterator<T> iterator() {
			return list.iterator();
		}

		@Override
		public Object[] toArray() {
			return list.toArray();
		}

		@Override
		public <E> E[] toArray(E[] a) {
			return list.toArray(a);
		}

		@Override
		public boolean add(T e) {
			boolean added = false;
			if(!contains(e))
				added = list.add(e);
			return added;
		}

		@Override
		public boolean remove(Object o) {
			return list.remove(o);
		}

		@Override
		public boolean containsAll(Collection<?> c) {
			return list.containsAll(c);
		}

		@Override
		public boolean addAll(Collection<? extends T> c) {
			boolean added = true;
			for(T element : c)
				added &= add(element);
			return added;
		}

		@Override
		public boolean retainAll(Collection<?> c) {
			return list.retainAll(c);
		}

		@Override
		public boolean removeAll(Collection<?> c) {
			return list.removeAll(c);
		}

		@Override
		public void clear() {
			list.clear();			
		}		
	}
}
