package tests;

import static org.junit.Assert.*;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import javax.activation.DataHandler;

import org.apache.chemistry.opencmis.client.api.CmisObject;
import org.apache.chemistry.opencmis.client.api.ObjectType;
import org.apache.chemistry.opencmis.client.api.Property;
import org.apache.chemistry.opencmis.client.api.Repository;
import org.apache.chemistry.opencmis.client.api.Session;
import org.apache.chemistry.opencmis.client.api.SessionFactory;
import org.apache.chemistry.opencmis.client.runtime.SessionFactoryImpl;
import org.apache.chemistry.opencmis.client.util.FileUtils;
import org.apache.chemistry.opencmis.commons.SessionParameter;
import org.apache.chemistry.opencmis.commons.definitions.PropertyDefinition;
import org.apache.chemistry.opencmis.commons.enums.BindingType;
import org.apache.chemistry.opencmis.commons.enums.Cardinality;
import org.apache.chemistry.opencmis.commons.enums.PropertyType;
import org.apache.chemistry.opencmis.commons.enums.Updatability;
import org.apache.chemistry.opencmis.commons.exceptions.CmisObjectNotFoundException;
import org.apache.commons.lang3.ArrayUtils;
import org.cmdbuild.common.Constants;

import org.cmdbuild.dms.MetadataGroup;
import org.cmdbuild.dms.DocumentDelete;
import org.cmdbuild.dms.DocumentDownload;
import org.cmdbuild.dms.DocumentSearch;
import org.cmdbuild.dms.DocumentUpdate;
import org.cmdbuild.dms.Metadata;
import org.cmdbuild.dms.StorableDocument;
import org.cmdbuild.dms.StoredDocument;
import org.cmdbuild.dms.cmis.CategoryLookupConverter;
import org.cmdbuild.dms.cmis.CmisDmsConfiguration;
import org.cmdbuild.dms.cmis.CmisDmsService;
import org.cmdbuild.dms.cmis.nuxeo.NuxeoForwardingService;
import org.cmdbuild.dms.exception.DmsError;
import org.junit.Before;
import org.junit.Test;

public class TestJFC implements CmisDmsConfiguration, CategoryLookupConverter {
	
	public TestJFC(){
		service = new CmisDmsService(this, this);
	}
	
	List<String> pathAsArray = Arrays.asList("ArchiMate","Technology","Node","Device","ClientDevice","OrganizationUnitClientDe","Id6022215");
	String pathName = "/ArchiMate/Technology/Node/Device/ClientDevice/OrganizationUnitClientDe/Id6022215";
	
	protected CmisDmsService service;
	Session directCmisSession;
	private static final SimpleDateFormat CMDBUILD_DATETIME_PARSING_FORMAT = new SimpleDateFormat(Constants.SOAP_ALL_DATES_PARSING_PATTERN);
	@Before
	public void prepare() {
		service = new CmisDmsService(this, this);
		
		SessionFactory factory = SessionFactoryImpl.newInstance();
		Map<String, String> parameter = new HashMap<String, String>();
	    
		parameter.put(SessionParameter.USER, getCmisUser());
		parameter.put(SessionParameter.PASSWORD,getCmisPassword() );

		parameter.put(SessionParameter.BINDING_TYPE, getCmisBindingType());
	     
	    parameter.put(SessionParameter.ATOMPUB_URL,  getCmisUrl());
	    parameter.put(SessionParameter.BROWSER_URL,  getCmisUrl());
	    parameter.put(SessionParameter.REPOSITORY_ID, "default");
	     
	    parameter.put(SessionParameter.LOCAL_FACTORY, getLocalServiceFactory());
	    
	    for(Entry<String,String> param : getSessionParameters().entrySet())
	    	parameter.put(param.getKey(), param.getValue());
	    	    
	    List<Repository> repositories = factory.getRepositories(parameter);
        Repository repository = repositories.get(0);
	      
        directCmisSession = repository.createSession();
        directCmisSession.getDefaultContext().setCacheEnabled(false);
	       
        assertNotNull(directCmisSession);
	       assertFalse(directCmisSession.getDefaultContext().isCacheEnabled());
		
	}

	@Test
	public void testCMISVersion() {
		assertEquals("CMIS_1_1",directCmisSession.getRepositoryInfo().getCmisVersion().toString());
		ObjectType typeDef = directCmisSession.getTypeDefinition("cmis:document");
		PropertyDefinition<?> propDef = typeDef.getPropertyDefinitions().get("cmis:secondaryObjectTypeIds");
		assertEquals(Updatability.READWRITE,propDef.getUpdatability());
	}
	 @Test
	   public void testCustomCmdbuildTypes() {
		 Session session = directCmisSession;
		   ObjectType res = null;

		   res = session.getTypeDefinition("facet:cmdbuild:imageFormat");
		   assertNotNull(res.getPropertyDefinitions().get("cmdbuild_if:imageFormat"));
		   assertNotNull(res.getPropertyDefinitions().get("cmdbuild_if:colored"));
		   assertEquals("org.cmdbuild.dms.alfresco",
				   res.getPropertyDefinitions().get("cmdbuild_if:colored").getLocalNamespace());

		   res=session.getTypeDefinition("facet:cmdbuild:displayable");
		   res = session.getTypeDefinition("facet:cmdbuild:displayable");
		   assertNotNull(res.getPropertyDefinitions().get("cmdbuild_d:width"));
		   assertNotNull(res.getPropertyDefinitions().get("cmdbuild_d:height"));
		   assertNotNull(res.getPropertyDefinitions().get("cmdbuild_d:sizeRatio"));
		   
		   
		   res = session.getTypeDefinition("facet:cmdbuild:documentStatistics");
		   assertNotNull(res.getPropertyDefinitions().get("cmdbuild_ds:characters"));
		   assertNotNull(res.getPropertyDefinitions().get("cmdbuild_ds:words"));
		   
		   res = session.getTypeDefinition("facet:cmdbuild:summary");
		   assertNotNull(res.getPropertyDefinitions().get("cmdbuild_s:summary"));
		   
		   res = session.getTypeDefinition("facet:cmdbuild:taggable");
		   assertNotNull(res.getPropertyDefinitions().get("cmdbuild_t:tags"));
		   
		   res = session.getTypeDefinition("facet:cmdbuild:creationDateTime");
		   assertNotNull(res.getPropertyDefinitions().get("cmdbuild_cdt:creationDate"));
		   assertNotNull(res.getPropertyDefinitions().get("cmdbuild_cdt:creationDateTime"));
	   }
	
	
	
	@Test
	public void uploadDocumentCategory() throws DmsError {
		try {
		FileUtils.delete(getCmisPath()+pathName, directCmisSession);
		} catch (CmisObjectNotFoundException e){
			
		}
		service.getTypeDefinitions();
		service.upload(new StorableDocument() {
			public Iterable<MetadataGroup> getMetadataGroups() { return null; }
			public List<String> getPath() {	return pathAsArray; }
			public String getClassName() { return "OrganizationUnitClientDe"; }
			public Long getCardId() {	return 6022215L;	}			
			public String getFileName() { return "test.txt"; }
			public String getDescription() { return "test"; }
			public String getCategory() { return "Document"; }
			public InputStream getInputStream() { return new ByteArrayInputStream("prova".getBytes());}
			public String getAuthor() { return "cmdbuild";}
		});
		CmisObject o = FileUtils.getObject(getCmisPath()+pathName+"/"+"test.txt", directCmisSession);
		assertNotNull(o);
		assertEquals("test.txt",o.getName());
		assertEquals("Note",o.getType().getId());
		FileUtils.printProperties(o);
		assertEquals("Document",o.getProperty("dc:nature").getValueAsString());
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:documentStatistics"));
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:taggable"));
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:summary"));
		
	}
	
	@Test

	public void uploadImageCategory() throws DmsError {
		this.uploadDocumentCategory();
	service.upload(new StorableDocument() {
		public Iterable<MetadataGroup> getMetadataGroups() { return null; }
		public List<String> getPath() {	return Arrays.asList("ArchiMate","Technology","Node","Device","ClientDevice","OrganizationUnitClientDe","Id6022215"); }
		public String getClassName() { return "OrganizationUnitClientDe"; }
		public Long getCardId() {	return 6022215L;	}			
		public String getFileName() { return "test.txt"; }
		public String getDescription() { return "test aaa bbb ccc"; }
		public String getCategory() { return "Image"; }
		public InputStream getInputStream() { return new ByteArrayInputStream("prova aaa bbb ccc".getBytes());}
		public String getAuthor() { return "cmdbuild";}
		});
		CmisObject o = FileUtils.getObject(getCmisPath()+pathName+"/"+"test.txt", directCmisSession);
		assertNotNull(o);
		assertEquals("test.txt",o.getName());
		o.refresh();
		//  Should this upload change the document category ?
		assertEquals("Image",o.getProperty("dc:nature").getValueAsString());
		assertFalse(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:documentStatistics"));
		
		
		assertFalse(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:documentStatistics"));
		assertFalse(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:summary"));
		
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:taggable"));
		
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:displayable"));
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:imageFormat"));
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:creationDateTime"));

	}
	
	@Test
	public void updateDescriptionAndMetadata() throws DmsError {
		uploadImageCategory();
	service.updateDescriptionAndMetadata(new DocumentUpdate() {
		public List<String> getPath() {	return Arrays.asList("ArchiMate","Technology","Node","Device","ClientDevice","OrganizationUnitClientDe","Id6022215"); }
		public String getClassName() { return "OrganizationUnitClientDe"; }
		public Long getCardId() {	return 6022215L;	}			
		public String getFileName() { return "test.txt"; }
		public String getDescription() { return "test 123"; }
		public String getCategory() { return null; }
		public String getAuthor() { return "admin"; }
		public Iterable<MetadataGroup> getMetadataGroups() {
			List<MetadataGroup> groups = new ArrayList<MetadataGroup>();
			groups.add(new MetadataGroup() {
				public String getName() {return "Displayable";}
				public Iterable<Metadata> getMetadata() {
					List<Metadata> metadatas = new ArrayList<Metadata>();
					metadatas.add(new Metadata() {
						public String getName() {return "cmdbuild_d:width";}
						public String getValue() {return Integer.toString(300);}
					});
					metadatas.add(new Metadata() {
						public String getName() {return "cmdbuild_d:height";}
						public String getValue() {return Integer.toString(200);}
					});
					metadatas.add(new Metadata() {
						public String getName() {return "cmdbuild_d:sizeRatio";}
						public String getValue() {return Double.toString(0.5);}
					});
					return metadatas;
				}
			});
			groups.add(new MetadataGroup() {
				public String getName() {return "Creation date/time";}
				public Iterable<Metadata> getMetadata() {
					List<Metadata> metadatas = new ArrayList<Metadata>();
					metadatas.add(new Metadata() {
						public String getName() {return "cmdbuild_cdt:creationDate";}
						public String getValue() {return CMDBUILD_DATETIME_PARSING_FORMAT.format(new Date());}
					});
					metadatas.add(new Metadata() {
						public String getName() {return "cmdbuild_cdt:creationDateTime";}
						public String getValue() {return CMDBUILD_DATETIME_PARSING_FORMAT.format(new Date());}
					});
					return metadatas;
				}
			});
			groups.add(new MetadataGroup() {
				public String getName() {return "Image format";}
				public Iterable<Metadata> getMetadata() {
					List<Metadata> metadatas = new ArrayList<Metadata>();
					metadatas.add(new Metadata() {
						public String getName() {return "cmdbuild_if:imageFormat";}
						public String getValue() {return "raw";}
					});
					metadatas.add(new Metadata() {
						public String getName() {return "cmdbuild_if:colored";}
						public String getValue() {return Boolean.toString(true);}
					});
					return metadatas;
				}
			});
			return groups; 
		}			
	});
	CmisObject o = FileUtils.getObject(getCmisPath()+pathName+"/"+"test.txt", directCmisSession);
	o.refresh();
	assertEquals("test 123",o.getPropertyValue("dc:description"));
	assertEquals("0.5",o.getPropertyValue("cmdbuild_d:sizeRatio").toString());
	assertEquals("200",o.getPropertyValue("cmdbuild_d:height").toString());
	assertEquals("300",o.getPropertyValue("cmdbuild_d:width").toString());
	assertEquals("raw",o.getPropertyValue("cmdbuild_if:imageFormat"));
	assertEquals(true,o.getPropertyValue("cmdbuild_if:colored"));
	assertNull("",o.getPropertyValue("cmdbuild_t:tags"));
	
	assertFalse(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:documentStatistics"));
	assertFalse(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:summary"));
	
	assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:taggable"));
	
	assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:displayable"));
	assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:imageFormat"));
	assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:creationDateTime"));

	
	FileUtils.printProperties(o);
	
	}

	@Test
	
	public void search() throws DmsError {
		String expectedPath = "/default-domain/UserWorkspaces/cmdbuild/cmdbuild/ArchiMate/Technology/Node/Device/ClientDevice/OrganizationUnitClientDe/Id6022215/test.txt";
		this.updateDescriptionAndMetadata();
		List<StoredDocument> result = service.search(new DocumentSearch() {			
			public List<String> getPath() {	return pathAsArray; }
			public String getClassName() { return "OrganizationUnitClientDe"; }
			public Long getCardId() {	return 6022215L;	}
		});
		assertEquals(1,result.size());
		StoredDocument searchResult = result.get(0);
		assertEquals("test.txt",searchResult.getName());
		assertEquals("test 123",searchResult.getDescription());
		assertEquals("Image",searchResult.getCategory());
		assertEquals("cmdbuild",searchResult.getAuthor());
		assertEquals(expectedPath,searchResult.getPath());
		Map<String,Iterable<Metadata>> mapResult = new HashMap<String,Iterable<Metadata>>();
		for (MetadataGroup mg : searchResult.getMetadataGroups()){
			mapResult.put(mg.getName(), mg.getMetadata());
			System.out.println(mg.getName());
		}
		assertTrue(mapResult.keySet().contains("Displayable"));
		assertTrue(mapResult.keySet().contains("Creation date/time"));
		assertTrue(mapResult.keySet().contains("Image format"));
		assertFalse(mapResult.keySet().contains("Document Statistics"));
		HashMap<String, String> metaMap = new   HashMap<String,String>();
		for (Metadata m : mapResult.get("Displayable")){
			metaMap.put(m.getName(), m.getValue());
		}
		assertEquals("300",metaMap.get("cmdbuild_d:width"));
		assertEquals("200",metaMap.get("cmdbuild_d:height"));
		assertEquals("0.5",metaMap.get("cmdbuild_d:sizeRatio"));
		metaMap = new   HashMap<String,String>();
		for (Metadata m : mapResult.get("Image format")){
			metaMap.put(m.getName(), m.getValue());
		}
		assertEquals("raw",metaMap.get("cmdbuild_if:imageFormat"));
		assertEquals("true",metaMap.get("cmdbuild_if:colored"));
		
		
		CmisObject o = FileUtils.getObject(getCmisPath()+pathName+"/"+"test.txt", directCmisSession);

		assertFalse(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:documentStatistics"));
		assertFalse(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:summary"));
		
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:taggable"));
		
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:displayable"));
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:imageFormat"));
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:creationDateTime"));

		
		
	}
	
	
	@Test
	public void download() throws DmsError, IOException {
		this.search();
		DataHandler downloaded = service.download(new DocumentDownload() {
			public List<String> getPath() {	return pathAsArray; }
			public String getClassName() { return "OrganizationUnitClientDe"; }
			public Long getCardId() {	return 6022215L;	}			
			public String getFileName() { return "test.txt"; }
		});
		assertNotNull(downloaded);
		assertEquals("test.txt",downloaded.getName());
		assertEquals("text/plain",downloaded.getContentType());
		Object content = downloaded.getContent();
		assertTrue(content instanceof String);
		assertEquals("prova aaa bbb ccc",content.toString());
	}
	@Test
	public void createTargetFolder() throws DmsError {
		try {
		FileUtils.delete(getCmisPath()+"/aaa/bbb/ccc/test/xxxxxx", directCmisSession);
		} catch (CmisObjectNotFoundException e){
			
		}
		service.create(new DocumentSearch() {
			public List<String> getPath() {	return Arrays.asList("aaa","bbb","ccc","test","xxxxxx"); }
			public String getClassName() { return null; }
			public Long getCardId() {	return null;	}
		});
		CmisObject o = FileUtils.getObject(getCmisPath()+"/aaa/bbb/ccc/test/xxxxxx" , directCmisSession);
		assertNotNull(o);
		assertEquals("Folder",o.getType().getDisplayName());
	}
	
	
	
	@Test
	public void copy() throws DmsError {
		this.search();
		this.createTargetFolder() ;
		service.copy(new StoredDocument() {
			public String getPath() { return pathName; }
			public String getName() { return "test.txt";}
		}, new DocumentSearch() {
			public List<String> getPath() {	return pathAsArray; }
			public String getClassName() { return "OrganizationUnitClientDe"; }
			public Long getCardId() {	return 6022215L;	}						
		},new DocumentSearch() {
			public List<String> getPath() {	return Arrays.asList("aaa","bbb","ccc","test","xxxxxx"); }
			public String getClassName() { return null; }
			public Long getCardId() {	return null;	}
		});
		CmisObject target = FileUtils.getObject(getCmisPath()+"/aaa/bbb/ccc/test/xxxxxx/test.txt" , directCmisSession);
		//pld doc is still here
		CmisObject source = FileUtils.getObject(getCmisPath()+pathName+"/"+"test.txt", directCmisSession);
		//target properties 
		
		CmisObject o = source;
		assertFalse(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:documentStatistics"));
		assertFalse(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:summary"));
		
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:taggable"));
		
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:displayable"));
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:imageFormat"));
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:creationDateTime"));

		o = target;
		assertFalse(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:documentStatistics"));
		assertFalse(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:summary"));
		
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:taggable"));
		
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:displayable"));
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:imageFormat"));
		assertTrue(o.getProperty("cmis:secondaryObjectTypeIds").getValues().contains("facet:cmdbuild:creationDateTime"));

		
		
		for (Property<?> property : source.getProperties()){
			String id = property.getId();
			String[] modifyableProperties = {
					"cmis:objectId",
					"cmis:creationDate",
					"cmis:lastModificationDate",
					"nuxeo:parentId",
					"cmis:versionSeriesId",
					"dc:modified",
					"dc:created",
					// Mmmm
					"cmis:isLatestVersion",
					"cmis:isLatestMajorVersion",
					"cmis:isMajorVersion",
					"cmis:versionLabel",
					"cmis:isVersionSeriesCheckedOut",
					"cmis:versionSeriesCheckedOutBy",
					"cmis:versionSeriesCheckedOutId",
					"nuxeo:isCheckedIn",
					"cmis:isPrivateWorkingCopy",
					"major_version"
					
			};
			
			if (! ArrayUtils.contains(modifyableProperties,id)){
				Object srcValue = source.getPropertyValue(id);
				Object tgtValue = target.getPropertyValue(id);
				assertEquals("Value are not equals for property " + id,srcValue,tgtValue);
			}
		}
	}
	
	
	@Test
	public void delete() throws DmsError {
		this.copy();
		service.delete(new DocumentDelete() {			
			public List<String> getPath() {	return pathAsArray; }
			public String getClassName() { return "OrganizationUnitClientDe"; }
			public Long getCardId() {	return 6022215L;	}			
			public String getFileName() { return "test.txt"; }			
		});
		boolean hasOccured = false;
		try {
			@SuppressWarnings("unused")
			CmisObject o = FileUtils.getObject(getCmisPath()+pathName+"/"+"test.txt", directCmisSession);
		} catch (CmisObjectNotFoundException e) {
			hasOccured=true;
		}
		assertTrue(hasOccured);
		}
	
	@Test
	public void create2() throws DmsError{
		//FileUtils.delete(, session);
		service.create(new DocumentSearch() {
			public List<String> getPath() {	return Arrays.asList("aaa","bbb","ccc","test","yyyyyy"); }
			public String getClassName() { return null; }
			public Long getCardId() {	return null;	}
		});
		CmisObject target = FileUtils.getObject(getCmisPath()+"/aaa/bbb/ccc/test/yyyyyy" , directCmisSession);
		assertNotNull(target);
	}
	@Test
	public void move() throws DmsError{
		boolean occured = false;
		FileUtils.delete(getCmisPath()+"/aaa", directCmisSession);
		try {
			FileUtils.getObject(getCmisPath()+"/aaa",directCmisSession);
			}  catch (CmisObjectNotFoundException e){
				occured=true;
			}
		assertTrue(occured);
		occured = false; 
		
		this.delete();
		this.create2();
		
		
		
		
		
		CmisObject source = FileUtils.getObject(getCmisPath()+"/aaa/bbb/ccc/test/xxxxxx/test.txt" , directCmisSession);
		
		List<Property<?>> srcProperties = new LinkedList<Property<?>>(source.getProperties());
		Map<String,List<?>> srcMapProperties = new HashMap<String,List<?>>();
		for (Property<?> p : srcProperties) {
			srcMapProperties.put(p.getId(), p.getValues());
		}
		
		service.move(new StoredDocument() {
			public String getPath() { return "/aaa/bbb/ccc/test/yyyyyy"; }
			public String getName() { return "test.txt";}
		}, new DocumentSearch() {
			public List<String> getPath() {	return Arrays.asList("aaa","bbb","ccc","test","xxxxxx"); }
			public String getClassName() { return null; }
			public Long getCardId() {	return null;	}						
		},new DocumentSearch() {
			public List<String> getPath() {	return Arrays.asList("aaa","bbb","ccc","test","yyyyyy"); }
			public String getClassName() { return null; }
			public Long getCardId() {	return null;	}
		});
		CmisObject target = FileUtils.getObject(getCmisPath()+"/aaa/bbb/ccc/test/yyyyyy/test.txt" , directCmisSession);
		

		
		try {
			FileUtils.getObject(getCmisPath()+"/aaa/bbb/ccc/test/xxxxxx/test.txt" , directCmisSession);
		} catch (CmisObjectNotFoundException e) {
			occured = true;
		}
		assertTrue(occured);
		FileUtils.printProperties(target);
		
		List<Property<?>> tgtProperties = new LinkedList<Property<?>>(target.getProperties());
		Map<String,List<?>> tgtMapProperties = new HashMap<String,List<?>>();
		for (Property<?> p : tgtProperties) {
			tgtMapProperties.put(p.getId(), p.getValues());
		}
		
		
		String[] modifyableProperties = {
				"nuxeo:parentId",
		};
		
		
		for (Property<?> property : srcProperties){
			String id = property.getId();
			
			if (! ArrayUtils.contains(modifyableProperties,id)){
				Object srcValue = property.getValues();
				List<?> tgtValue = tgtMapProperties.get(id);
				System.out.println(id + "---- " + srcValue +" -----> "+tgtValue);
				assertEquals("Values are not equals for property " + id,srcValue,tgtValue);
				
			
			}
		}
		
	}
	
	@Test
	public void delete2() throws DmsError{
		boolean occured = false; 
		
		move();
		service.delete(new DocumentSearch() {
			public List<String> getPath() {	return Arrays.asList("aaa"); }
			public String getClassName() { return null; }
			public Long getCardId() {	return null;	}
		});
		
		//FileUtils.getObject(getRepositoryWSPath()+getRepositoryApp()+"/aaa",directCmisSession);
		
		try {
		FileUtils.getObject(getCmisPath()+"/aaa",directCmisSession);
		}  catch (CmisObjectNotFoundException e){
			occured=true;
		}
		assertTrue(occured);
	}

	
	
	@Test
	public void testNuxeoCustomModel(){
		Session session = directCmisSession;
    	ObjectType res = session.getTypeDefinition("Folder");
    	assertNotNull(res);
    	for (ObjectType child : res.getChildren()){
    		System.out.println(child.getLocalName());
    	}
    	res = session.getTypeDefinition("cmis:secondary");
    	assertNotNull(res);
    	for (ObjectType child : res.getChildren()){
    		System.out.println(child.getLocalName());
    		System.out.println(child.getLocalNamespace());
    	}
    	
    	
    	res = session.getTypeDefinition("cmis:document");
    	for (ObjectType child : res.getChildren()){
    		System.out.println(child.getLocalName());
    		System.out.println(child.getLocalNamespace().length());
    	}
    	res = session.getTypeDefinition("facet:cmdbuild:displayable");
    	assertEquals("cmdbuild:displayable",res.getLocalName());
    	assertEquals("http://ns.nuxeo.org/cmis/facet/",res.getLocalNamespace());
    	assertEquals("facet:cmdbuild:displayable",res.getDisplayName());
    	assertEquals("facet:cmdbuild:displayable",res.getId());
    	Map<String, PropertyDefinition<?>> pd = res.getPropertyDefinitions();
    	assertNotNull(pd);
    	PropertyDefinition<?> width = pd.get("cmdbuild_d:width");
    	assertNotNull(width);
    	assertEquals("cmdbuild_d:width",width.getLocalName());
    	assertNull(width.getLocalNamespace());
    	assertEquals(Cardinality.SINGLE,width.getCardinality());
    	assertFalse(width.isRequired());
    	res = session.getTypeDefinition("facet:cmdbuild:imageFormat");
    	pd = res.getPropertyDefinitions();
    	PropertyDefinition<?> format = pd.get("cmdbuild_if:imageFormat");
    	assertNotNull(format);
    	assertNull(format.getLocalNamespace());
    	assertEquals(PropertyType.STRING,format.getPropertyType());
    	for (Entry<String, PropertyDefinition<?>>entry : pd.entrySet()){
    		System.out.println(entry.getKey() + entry.getValue().getCardinality());
    	}
	}
	
	@Override
	public boolean isEnabled() {
		return true;
	}

	@Override
	public String getCmisUrl() {
		return "http://localhost:38080/browser";
	}

	@Override
	public String getCmisUser() {
		return "cmdbuild";
	}

	@Override
	public String getCmisPassword() {
		return "cmdbuild";
	}

	@Override
	public String getCmdbuildCategory() {
		return "CmdbuildCategory";
	}

	@Override
	public String getCmisPath() {
		return "/default-domain/UserWorkspaces/cmdbuild";
	}

	@Override
	public String getAlfrescoCustomUri() {
		return "org.cmdbuild.dms.alfresco";
	}
	
	
	@Override
	public String getCustomModelFileContent() {
		StringBuilder  stringBuilder = new StringBuilder();
		try {
		    String ls = System.getProperty("line.separator");
			BufferedReader reader = new BufferedReader(new FileReader ("../cmdbuild-dms-cmis-nuxeo/cmisNuxeoCustomModel.xml"));
		    String line = null;
		    while( ( line = reader.readLine() ) != null ) {
		        stringBuilder.append( line );
		        stringBuilder.append( ls );
		    }					
		    reader.close();
		} catch (Exception e) {
			e.printStackTrace();
		}		
		return stringBuilder.toString();
	}

	@Override
	public String getMetadataAutocompletionFileContent() {
		return null;
	}

	@Override
	public String getCmisModelType() {
		return "custom";
	}
	
	public String getCmisBindingType() {
		return BindingType.LOCAL.value();
	}

	public String getLocalServiceFactory() {
		return "org.cmdbuild.dms.cmis.nuxeo.NuxeoServiceFactory";
	}
	
	public Map<String, String> getSessionParameters() {
		HashMap<String, String> params = new HashMap<String, String>();
	    params.put(NuxeoForwardingService.NUXEO_AUTOMATION_URL, "http://localhost/nuxeo/api/v1/automation");	
	    params.put(NuxeoForwardingService.NUXEO_AUTOMATION_USER, getCmisUser());
	    params.put(NuxeoForwardingService.NUXEO_AUTOMATION_PASSWORD,	getCmisPassword());
	    return params;
	}

	@Override
	public void addListener(ChangeListener listener) {
	}

	@Override
	public String getService() {
		return "cmis";
	}

	@Override
	public Object toCmis(String value) {
		return value;
	}

	@Override
	public String fromCmis(Object value) {
		return value != null ? value.toString() : null;
	}
}
