package it.unibo.cmdb.archimate;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAnyElement;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlSchemaType;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.ecore.EClass;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSSerializer;

import com.archimatetool.model.IArchimatePackage;
import com.archimatetool.model.util.ArchimateModelUtils;

@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement(name = "cmdb")
public class CMDBConfiguration {
	
	public static final String PROPERTY_SAVE = "save";
	
	@XmlAccessorType(XmlAccessType.FIELD)
	public static class IdRule {
		
		@XmlElement(required = true)
	    protected String type;
		
		@XmlElement(required = true)
	    @XmlSchemaType(name = "anyURI")
	    protected String mdrId;
		
		@XmlElement(required = true)
	    @XmlSchemaType(name = "anyURI")
	    protected String localId;
		
		public String getType() {
			return type;
		}

		public void setType(String type) {
			this.type = type;
		}

		public String getMdrId() {
			return mdrId;
		}

		public void setMdrId(String mdrId) {
			this.mdrId = mdrId;
		}

		public String getLocalId() {
			return localId;
		}

		public void setLocalId(String localId) {
			this.localId = localId;
		}
	}
	
	@XmlTransient
	protected PropertyChangeSupport listeners;
    
	@XmlTransient
	protected URI uri;
	
	@XmlAnyElement
	protected Element query;
	
    @XmlElement(required = true)
    @XmlSchemaType(name = "anyURI")
    protected String archi2cmdb;
    
    @XmlElement(required = true)
    @XmlSchemaType(name = "anyURI")    
    protected String cmdb2archi;
    
    @XmlElement(required = true)    
    protected QName queryService;
    
    @XmlElement(required = true)    
    protected QName queryPort;
    
    @XmlElement(required = true)
    @XmlSchemaType(name = "anyURI")
    protected String queryWSDL;
    
    @XmlElement(required = true)
    protected QName registrationService;
    
    @XmlElement(required = true)    
    protected QName registrationPort;
    
    @XmlElement(required = true)
    @XmlSchemaType(name = "anyURI")
    protected String registrationWSDL;
    
    @XmlElement(required = true)
    protected boolean dedupRelationships;
    
    @XmlElement(required = true)
    protected boolean detectDerivedRelationships;
    
    @XmlElement(name="idRule", required = true)
    protected List<IdRule> idRules;
    
    protected CMDBConfiguration() {
    	listeners = new PropertyChangeSupport(this);
    }
    
    public URI getUri() {
		return uri;
	}

	public void setUri(URI uri) {
		this.uri = uri;
	}
        
	public Element getQuery() {
		return query;
	}

	public void setQuery(Element query) {
		this.query = query;
	}

	public String getArchi2cmdb() {
		return archi2cmdb;
	}

	public void setArchi2cmdb(String archi2cmdb) {
		this.archi2cmdb = archi2cmdb;
	}

	public String getCmdb2archi() {
		return cmdb2archi;
	}

	public void setCmdb2archi(String cmdb2archi) {
		this.cmdb2archi = cmdb2archi;
	}

	public QName getQueryService() {
		return queryService;
	}

	public void setQueryService(QName queryService) {
		this.queryService = queryService;
	}

	public QName getQueryPort() {
		return queryPort;
	}

	public void setQueryPort(QName queryPort) {
		this.queryPort = queryPort;
	}

	public String getQueryWSDL() {
		return queryWSDL;
	}

	public void setQueryWSDL(String queryWSDL) {
		this.queryWSDL = queryWSDL;
	}

	public QName getRegistrationService() {
		return registrationService;
	}

	public void setRegistrationService(QName registrationService) {
		this.registrationService = registrationService;
	}

	public QName getRegistrationPort() {
		return registrationPort;
	}

	public void setRegistrationPort(QName registrationPort) {
		this.registrationPort = registrationPort;
	}

	public String getRegistrationWSDL() {
		return registrationWSDL;
	}

	public void setRegistrationWSDL(String registrationWSDL) {
		this.registrationWSDL = registrationWSDL;
	}

	public boolean isDedupRelationships() {
		return dedupRelationships;
	}

	public void setDedupRelationships(boolean dedupRelationships) {
		this.dedupRelationships = dedupRelationships;
	}

	public boolean isDetectDerivedRelationships() {
		return detectDerivedRelationships;
	}

	public void setDetectDerivedRelationships(boolean detectDerivedRelationships) {
		this.detectDerivedRelationships = detectDerivedRelationships;
	}

	public List<IdRule> getIdRules() {
		return idRules;
	}

	public void setIdRules(List<IdRule> idRules) {
		this.idRules = idRules;
	}
	
	public void addPropertyChangeListener(PropertyChangeListener listener) {
        listeners.addPropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        listeners.removePropertyChangeListener(listener);
    }
    
    public void firePropertyChange(Object source, String prop, Object oldValue, Object newValue) {
        listeners.firePropertyChange(new PropertyChangeEvent(source, prop, oldValue, newValue));
    }

	public static CMDBConfiguration load(URI uri) throws Exception {
		DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
		documentBuilderFactory.setNamespaceAware(true);
		JAXBContext context = JAXBContext.newInstance(CMDBConfiguration.class);
		CMDBConfiguration configuration = null;
		try {
			Document configDoc = documentBuilderFactory.newDocumentBuilder().parse(uri.toString());
			Unmarshaller unmarshaller = context.createUnmarshaller();
			configuration = (CMDBConfiguration)unmarshaller.unmarshal(configDoc);
			configuration.uri = uri;
		} catch (Exception e) {
			IStatus status = new Status(IStatus.ERROR, CMDBPlugin.PLUGIN_ID, e.getMessage(), e);
			CMDBPlugin.INSTANCE.getLog().log(status);
			configuration = getDefaults();
			configuration.uri = uri;
			configuration.save();			
		}
		return configuration;
	}
	
	public void save() throws JAXBException, ParserConfigurationException {
		DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
		documentBuilderFactory.setNamespaceAware(true);
		Document configDoc = documentBuilderFactory.newDocumentBuilder().newDocument();
		JAXBContext context = JAXBContext.newInstance(CMDBConfiguration.class);		
		Marshaller marshaller = context.createMarshaller();
		marshaller.marshal(this, configDoc);
		DOMImplementationLS domImplementation = (DOMImplementationLS) configDoc.getImplementation();
		LSSerializer lsSerializer = domImplementation.createLSSerializer();
		if(lsSerializer.getDomConfig().canSetParameter("format-pretty-print", Boolean.TRUE))
			lsSerializer.getDomConfig().setParameter("format-pretty-print", Boolean.TRUE);
		lsSerializer.writeToURI(configDoc, uri.toString());
		firePropertyChange(this, PROPERTY_SAVE, null, this);
	}	
	
	public static CMDBConfiguration getDefaults() {
		CMDBConfiguration configuration = new CMDBConfiguration();
		
		File userDataFolder = CMDBPlugin.INSTANCE.getUserDataFolder();
		File archi2cmdb = new File(userDataFolder, "archi2cmdbuild.atl");
		saveResource("/model/archi2cmdbuild.atl", archi2cmdb);
	    File cmdb2archi = new File(userDataFolder, "cmdbuild2archi.atl");
		saveResource("/model/cmdbuild2archi.atl", cmdb2archi);
		
		configuration.setQuery(getDefaultQuery());
		configuration.setArchi2cmdb(archi2cmdb.toURI().toString());
		configuration.setCmdb2archi(cmdb2archi.toURI().toString());
		configuration.setQueryService(new QName("http://schemas.dmtf.org/cmdbf/1/tns/query", "CMDBfQueryImplService"));
		configuration.setQueryPort(new QName("http://schemas.dmtf.org/cmdbf/1/tns/query", "CMDBfQueryImplPort"));
		configuration.setQueryWSDL("http://localhost:8080/cmdbuild/services/soap/CMDBfQuery?wsdl");
		configuration.setRegistrationService(new QName("http://schemas.dmtf.org/cmdbf/1/tns/registration", "CMDBfRegistrationImplService"));
		configuration.setRegistrationPort(new QName("http://schemas.dmtf.org/cmdbf/1/tns/registration", "CMDBfRegistrationImplPort"));
		configuration.setRegistrationWSDL("http://localhost:8080/cmdbuild/services/soap/CMDBfRegistration?wsdl");
		configuration.setDedupRelationships(true);
		configuration.setDetectDerivedRelationships(true);
		configuration.setIdRules(getDefaultIdRules());
		
		return configuration;
    }

	private static Element getDefaultQuery() {
    	StringBuilder xml = new StringBuilder();
	    String classNS = "http://www.cmdbuild.org/class";
	    String domainNS = "http://www.cmdbuild.org/domain";
	    String recordTypeFilter = "      <cmdbf:recordType localName=\"%s\" namespace=\"%s\"/>\n"; 
	    xml.append("<cmdbf:query xmlns:cmdbf=\"http://schemas.dmtf.org/cmdbf/1/tns/serviceData\">\n");
	    xml.append("  <cmdbf:itemTemplate id=\"items\">\n");
	    xml.append("    <cmdbf:recordConstraint>\n");
	    for(EClass eClass : ArchimateModelUtils.getApplicationClasses())
	    	xml.append(String.format(recordTypeFilter, eClass.getName(), classNS));
	    for(EClass eClass : ArchimateModelUtils.getBusinessClasses())
	    	xml.append(String.format(recordTypeFilter, eClass.getName(), classNS));
	    for(EClass eClass : ArchimateModelUtils.getConnectorClasses())
	    	xml.append(String.format(recordTypeFilter, eClass.getName(), classNS));
	    for(EClass eClass : ArchimateModelUtils.getImplementationMigrationClasses())
	    	xml.append(String.format(recordTypeFilter, eClass.getName(), classNS));
	    for(EClass eClass : ArchimateModelUtils.getMotivationClasses())
	    	xml.append(String.format(recordTypeFilter, eClass.getName(), classNS));
	    for(EClass eClass : ArchimateModelUtils.getTechnologyClasses())
	    	xml.append(String.format(recordTypeFilter, eClass.getName(), classNS));
    	xml.append(String.format(recordTypeFilter, "ArchimateDiagramModel", classNS));
	    xml.append("    </cmdbf:recordConstraint>\n");
	    xml.append("  </cmdbf:itemTemplate>\n");
	    xml.append("  <cmdbf:relationshipTemplate id=\"relationships\">\n");
	    xml.append("    <cmdbf:recordConstraint>\n");
	    for(EClass eClass : ArchimateModelUtils.getRelationsClasses())
	    	xml.append(String.format(recordTypeFilter, eClass.getName(), domainNS));
    	xml.append(String.format(recordTypeFilter, "DiagramModelReference", domainNS));
	    xml.append("    </cmdbf:recordConstraint>\n");
	    xml.append("  </cmdbf:relationshipTemplate>\n");
	    xml.append("</cmdbf:query>\n");
	    
	    return ModelUtils.parseXml(xml.toString());
    }
    
    private static List<IdRule> getDefaultIdRules() {
	    List<IdRule> rules = new ArrayList<CMDBConfiguration.IdRule>();
	    
    	IdRule relationshipRule = new IdRule();
    	relationshipRule.setType(IArchimatePackage.eINSTANCE.getRelationship().getName());
    	relationshipRule.setMdrId("");
    	relationshipRule.setLocalId("{Type}#{Source}#{Target}");
    	rules.add(relationshipRule);
    	
    	IdRule diagramRule = new IdRule();
    	diagramRule.setType(IArchimatePackage.eINSTANCE.getArchimateDiagramModel().getName());
    	diagramRule.setMdrId("CMDBuild:{\"regex\":\"(\\\\w+)#(.+)\",\"query\":\"from ArchiMate where Code={g2}\",\"type\":\"{g1}\"}");
    	diagramRule.setLocalId("{CMDB-Type}#{Name}");
    	rules.add(diagramRule);
    	
    	IdRule elementRule = new IdRule();
    	elementRule.setType(IArchimatePackage.eINSTANCE.getArchimateElement().getName());
    	elementRule.setMdrId("CMDBuild:{\"regex\":\"(\\\\w+)#(.+)\",\"query\":\"from ArchiMate where Code={g2}\",\"type\":\"{g1}\"}");
    	elementRule.setLocalId("{CMDB-Type}#{Name}");
    	rules.add(elementRule);
    	
    	return rules;
	}
    
	private static void saveResource(String resourceName, File resource) {
		try {
			if(!resource.exists()) {
				InputStream input = CMDBConfiguration.class.getResourceAsStream(resourceName);
				if(input != null) {
				    OutputStream output = new FileOutputStream(resource);
				    byte[] buffer = new byte[1024];
				    int len;
				    while ((len = input.read(buffer)) != -1) {
				        output.write(buffer, 0, len);
				    }
				    input.close();
				    output.close();
				}
			}
		} catch(Exception e) {
			IStatus status = new Status(IStatus.ERROR, CMDBPlugin.PLUGIN_ID, e.getMessage(), e);
			CMDBPlugin.INSTANCE.getLog().log(status); 			
		}
    }

}
