package it.unibo.cmdb.archimate;

import java.util.Iterator;
import java.util.Map;

import javax.xml.bind.JAXBException;
import javax.xml.datatype.XMLGregorianCalendar;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EContentAdapter;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xml.type.XMLTypePackage;

import com.archimatetool.model.FolderType;
import com.archimatetool.model.IArchimateElement;
import com.archimatetool.model.IArchimateModel;
import com.archimatetool.model.IFolder;
import com.archimatetool.model.IIdentifier;
import com.archimatetool.model.IProperty;
import com.archimatetool.model.IRelationship;
import com.archimatetool.model.impl.ArchimateElement;
import com.archimatetool.model.impl.ArchimateFactory;
import com.archimatetool.model.impl.DiagramModel;
 
public class CMDBContentAdapter extends EContentAdapter {
	
	public final static String DELETED_FOLDER = "DELETED";
	private final static String LASTMODIFIED_PROPERTY = "LastModified";
	private final static String MDRID_PROPERTY = "MdrId";
	private final static String LOCALID_PROPERTY = "LocalId";
	
	private IArchimateModel model;
	private IFolder deletedFolder;
	private IFolder derivedFolder;
	private boolean trashDisabled;
	private CMDBMetadata metadata;
	
	public CMDBContentAdapter(IArchimateModel model) {
		this.model = model;
		deletedFolder = getDeletedFolder(model);
		derivedFolder = getDerivedFolder(model);
		metadata = new CMDBMetadata();
		IProperty metadataProperty = model.getMetadata().getEntry(CMDBPlugin.PLUGIN_ID);
		if(metadataProperty != null) {
			try {
				metadata.load(metadataProperty.getValue());
			} catch (JAXBException e) {
				LogUtils.logException(e);
			}
		}
	}
	
	public void saveMetadata() {
		String value = null;
		try {
			value = metadata.save();
		} catch (JAXBException e) {
			LogUtils.logException(e);
		}
		model.getMetadata().addEntry(CMDBPlugin.PLUGIN_ID, value);
	}
	
	public void mergeMetadata(IArchimateModel otherModel) {
		CMDBMetadata newMetadata = new CMDBMetadata();
		CMDBMetadata otherMetadata = otherModel!=null ? getAdapter(otherModel).metadata : null;
		
		TreeIterator<EObject> contentIterator = model.eAllContents();
		while(contentIterator.hasNext()) {
			EObject eObject = contentIterator.next();
			if(eObject instanceof IIdentifier) {
				String id = ((IIdentifier)eObject).getId();
				Map<String, String> map = null;
				if(otherMetadata != null)
					map = otherMetadata.getMetadata(id);
				if(map == null)
					map = metadata.getMetadata(id);
				if(map!=null)
					newMetadata.putMetadata(id, map);
			}
		}
		metadata = newMetadata;
		saveMetadata();
	}
	
	public void disableTrash(boolean disabled) {
		trashDisabled = disabled;
	}
	
	public IFolder getDeletedFolder() {
		return deletedFolder;
	}
	
	public IFolder getDerivedFolder() {
		return derivedFolder; 
	}
	
	private IFolder getDeletedFolder(IArchimateModel model) {
		IFolder deletedFolder = null;
		Iterator<IFolder> iterator = model.getFolders().iterator();
    	while(iterator.hasNext() && deletedFolder==null) {
    		IFolder current = iterator.next();
    		if(current.getName().equals(DELETED_FOLDER))
    			deletedFolder = current;
    	}
    	if(deletedFolder == null) {
    		deletedFolder = ArchimateFactory.eINSTANCE.createFolder();
    		deletedFolder.setName(DELETED_FOLDER);
    		model.getFolders().add(deletedFolder);
    	}
    	return deletedFolder;
	}
	
	private IFolder getDerivedFolder(IArchimateModel model) {
		IFolder derivedFolder = null;		
		derivedFolder = model.getFolder(FolderType.DERIVED);
		if(derivedFolder == null)
			derivedFolder = model.addDerivedRelationsFolder();
		return derivedFolder; 
	}
	
	public static CMDBContentAdapter getAdapter(Notifier notifier) {
		CMDBContentAdapter cmdbContentAdapter = null;
		Iterator<Adapter> iterator = notifier.eAdapters().iterator();
		while(cmdbContentAdapter==null && iterator.hasNext()) {
			Adapter adapter = iterator.next();
			if(adapter instanceof CMDBContentAdapter)
				cmdbContentAdapter = (CMDBContentAdapter)adapter;
		}
		if(cmdbContentAdapter == null) {
			IArchimateModel model = null;
			if(notifier instanceof EObject){
				EObject eObject = (EObject)notifier;
				while(model==null && eObject!=null) {
					if(eObject instanceof IArchimateModel)
						model = (IArchimateModel)eObject;
					eObject = eObject.eContainer();
				}
			}
			if(model != null) {
				cmdbContentAdapter = new CMDBContentAdapter(model);
				model.eAdapters().add(cmdbContentAdapter);
			}
		}
		return cmdbContentAdapter;
	}

	@Override
	public void notifyChanged(Notification notification) {
		super.notifyChanged(notification);
		switch(notification.getEventType()){
			case Notification.REMOVE: {
				if(deletedFolder != notification.getNotifier()) {
					if(notification.getOldValue() instanceof IIdentifier) {
						IIdentifier deleted = (IIdentifier)notification.getOldValue();						
						if(!trashDisabled && (deleted instanceof ArchimateElement || deleted instanceof DiagramModel || deleted instanceof IFolder)) {		            	
							deletedFolder.getElements().add((EObject)notification.getOldValue());
						}
						else {
							metadata.remove(deleted.getId());
							saveMetadata();
						}
					}
				}
				break;
			}
			case Notification.REMOVING_ADAPTER: {
				if(model == notification.getNotifier())
					saveMetadata();
				break;
			}
		}
    }

	public boolean isDeleted(EObject eObject) {
		boolean deleted = false;
		if(eObject instanceof IArchimateElement && ((IArchimateElement)eObject).getArchimateModel()==null)
			deleted = true;
		if(deletedFolder != null && eObject != deletedFolder) {
			while(!deleted && eObject!=null){
				deleted = eObject == deletedFolder;
				eObject = eObject.eContainer();
			}
		}
		return deleted;
	}
	
	public boolean isDerived(IRelationship relationship) {
		boolean detectDeivedRelationships = CMDBPlugin.INSTANCE.getConfiguration().isDetectDerivedRelationships();
        boolean derived = false;
		if(derivedFolder != null && detectDeivedRelationships) {
			EObject eObject = relationship;
			while(!derived && eObject!=null){
				derived = eObject == derivedFolder;
				eObject = eObject.eContainer();
			}
		}							
		return derived;
	}

	public void emptyDeleted() {
		deletedFolder.getElements().clear();
	}
	
	public String getMdrId(IIdentifier element){
		return metadata.get(element.getId(), MDRID_PROPERTY);
	}
	
	public void setMdrId(IIdentifier element, String value) {
		metadata.put(element.getId(), MDRID_PROPERTY, value);
	}
	
	public String getLocalId(IIdentifier element){
		return metadata.get(element.getId(), LOCALID_PROPERTY);
	}
	
	public void setLocalId(IIdentifier element, String value) {
		metadata.put(element.getId(), LOCALID_PROPERTY, value);
	}
	
	public XMLGregorianCalendar getLastModified(IIdentifier element){
		XMLGregorianCalendar value = null;
		String propertyValue = metadata.get(element.getId(), LASTMODIFIED_PROPERTY);
		if(propertyValue != null && !propertyValue.isEmpty())
			value = (XMLGregorianCalendar)EcoreUtil.createFromString(XMLTypePackage.eINSTANCE.getDateTime(), propertyValue);
		return value;
	}
	
	public void setLastModified(IIdentifier element, XMLGregorianCalendar value) {
		String propertyValue = null;
		if(value != null)
			propertyValue = EcoreUtil.convertToString(XMLTypePackage.eINSTANCE.getDateTime(), value);
		metadata.put(element.getId(), LASTMODIFIED_PROPERTY, propertyValue);
	}
}
