package it.unibo.cmdb.archimate;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.progress.IProgressService;

import com.archimatetool.editor.model.IEditorModelManager;
import com.archimatetool.editor.model.IModelExporter;
import com.archimatetool.model.IArchimateModel;
import com.archimatetool.model.IIdentifier;

import it.unibo.cmdb.archimate.ModelTransformer.TransformationResult;
import it.unibo.cmdb.archimate.model.transformationUtil.TransformationUtil;
import it.unibo.cmdb.archimate.model.transformationUtil.TransformationUtilFactory;
import it.unibo.cmdb.cmdbf.client.CMDBClient;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.MdrScopedIdType;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.Model;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ModelUpdateInstanceResponse;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.ModelUpdateResponse;
import it.unibo.cmdb.cmdbf.client.model.cmdbf.RegisterInstanceResponseType;

public class CMDBExporter implements IModelExporter {
    
	private class ExportJob implements IRunnableWithProgress {
		
		private IArchimateModel model;
		private boolean ignoreMetadata;
		private IStatus status;
		
		public ExportJob(IArchimateModel model, boolean ignoreMetadata) {
			this.model = model;
			this.ignoreMetadata = ignoreMetadata;
		}
		
		public IStatus getStatus() {
			return status;
		}

		@Override
		public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
			try {
				status = doExport(monitor, model, ignoreMetadata);
			} catch (Throwable e) {
				LogUtils.logException(e);
				throw new InvocationTargetException(e, e.getMessage());
			}
		}
	}

    @Override
    public void export(IArchimateModel model) throws IOException {
    	CMDBDialog dialog = new CMDBDialog(Display.getCurrent().getActiveShell(), model, false);
    	if(dialog.open() != Window.OK)
    		return;
    	
    	try {
	        boolean ignoreMetadata = dialog.getIgnoreMetadata();
	        model = dialog.getModel();
	        
	        if(model.eResource() == null)
	        	IEditorModelManager.INSTANCE.saveModel(model);
	        
	        if(model.eResource() != null) {	        
		        CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(model);
		        adapter.saveMetadata();
		        
		        IEditorModelManager.INSTANCE.closeModel(model);
				
		        ExportJob exportJob = new ExportJob(model, ignoreMetadata);
				IProgressService progressService = CMDBPlugin.INSTANCE.getWorkbench().getProgressService();
				progressService.run(true, true, exportJob);
				ModelUtils.makeModelDirty(model);
							
				if(exportJob.getStatus().isOK())
					adapter.emptyDeleted();
				else
					ErrorDialog.openError(dialog.getShell(), this.getClass().getName(), "CMDB export failed", exportJob.getStatus());
	        }
    	} catch (Exception e) {
    		LogUtils.logException(e);
			Throwable cause = null;
    		if(e instanceof InvocationTargetException && ((InvocationTargetException)e).getCause()!=null)
    			cause = ((InvocationTargetException)e).getCause();
    		else
    			cause = e;
			IStatus status = new Status(IStatus.ERROR, CMDBPlugin.PLUGIN_ID, cause.getMessage(), cause);
			ErrorDialog.openError(dialog.getShell(), this.getClass().getName(), "CMDB export aborted", status);
		}
    	finally{
    		IEditorModelManager.INSTANCE.openModel(model);
    	}
    }
    
    private IStatus doExport(IProgressMonitor monitor, IArchimateModel model, boolean ignoreMetadata) throws Exception {
    	monitor.beginTask("Export CMDB model", 10);
    	
    	boolean dedupRelationships = CMDBPlugin.INSTANCE.getConfiguration().isDedupRelationships();
        boolean detectDeivedRelationships = CMDBPlugin.INSTANCE.getConfiguration().isDetectDerivedRelationships();
                        
        CMDBClient cmdbClient = null;
        if(!monitor.isCanceled()) {
			monitor.subTask("Initialize CMDBf client");
			cmdbClient = CMDBPlugin.INSTANCE.getCmdbClient();
			monitor.worked(1);
			if(cmdbClient == null)
				monitor.setCanceled(true);
        }
        
        ModelTransformer transformer = null;        
        if(!monitor.isCanceled()) {
			monitor.subTask("Initialize ATL");
			transformer = CMDBPlugin.INSTANCE.getModelTransformer();
			monitor.worked(1);
			if(transformer == null)
				monitor.setCanceled(true);
        }
                
        DiagramSerializer diagramSerializer = new DiagramSerializer();
		TransformationUtil util = TransformationUtilFactory.eINSTANCE.createTransformationUtil();
		util.setCmdbfClient(cmdbClient);
		util.setDiagramSerializer(diagramSerializer);
		if(!monitor.isCanceled()) {
        	monitor.subTask("Prepare model");
        	if(detectDeivedRelationships) {
        		ModelClosure closure = new ModelClosure(model);
        		closure.transitiveReduction();
        	}
        	monitor.worked(1);
        }        
        
        Model cmdbModel = null;
        Map<MdrScopedIdType, IIdentifier> idMap = null;
        if(!monitor.isCanceled()) {
			monitor.subTask("Apply ATL transtormation");
	        monitor.worked(1);
			TransformationResult transformationResult = transformer.exportModel(model.eResource(), util);			
	        Resource outModel = transformationResult.getResource();
	        idMap = ModelUtils.updateCMDBElements(transformationResult.getTrace(), ignoreMetadata);
	        monitor.worked(2);
	        
	        EList<EObject> contents = outModel.getContents();
			if(contents != null) {
				for(EObject eObject : contents) {
					if(eObject instanceof Model) {
						cmdbModel = (Model)eObject;
						cmdbModel.setMdrId(cmdbClient.getMdrId());
						if(dedupRelationships)
							ModelUtils.dedupCMDBRelationships(cmdbModel.getRelationship());
					}
				}
			}
        }
	    
		MultiStatus status = new MultiStatus(CMDBPlugin.PLUGIN_ID, 0, this.getClass().getName(), null);
		if(!monitor.isCanceled() && cmdbModel != null) {
			monitor.subTask("Execute CMDBf registration");
			ModelUpdateResponse updateResponse = cmdbClient.update(cmdbModel);
			XMLGregorianCalendar lastModified = DatatypeFactory.newInstance().newXMLGregorianCalendar(new GregorianCalendar());
			monitor.worked(4);
	        	        
	        List<ModelUpdateInstanceResponse> failed = new ArrayList<ModelUpdateInstanceResponse>();
			if(updateResponse != null) {
				for(ModelUpdateInstanceResponse instanceResponse :  updateResponse.getInstanceResponse()){
					if(instanceResponse.getDeclined() != null) {
						failed.add(instanceResponse);
					}
					else if(instanceResponse instanceof RegisterInstanceResponseType) {
						RegisterInstanceResponseType registerResponse = (RegisterInstanceResponseType)instanceResponse;
						if(registerResponse.getAccepted() != null) {
							if(idMap != null && registerResponse.getInstanceId()!=null) {
								IIdentifier archiElement = idMap.get(registerResponse.getInstanceId());
								Iterator<MdrScopedIdType> iterator = registerResponse.getAccepted().getAlternateInstanceId().iterator();
								while(archiElement==null && iterator.hasNext())
									archiElement = idMap.get(iterator.next());
								if(archiElement != null) {
									CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(archiElement);
									adapter.setMdrId(archiElement, registerResponse.getInstanceId().getMdrId());
									adapter.setLocalId(archiElement, registerResponse.getInstanceId().getLocalId());
									adapter.setLastModified(archiElement, lastModified);
								}
							}
						}
					}
				}
				CMDBContentAdapter.getAdapter(model).saveMetadata();				
			}
			
			Model diagramModel = cmdbClient.newModel();
			if(diagramSerializer.updateDiagrams(diagramModel)) {
				ModelUpdateResponse diagramUpdateResponse = cmdbClient.update(diagramModel);
				if(diagramUpdateResponse != null) {
					for(ModelUpdateInstanceResponse instanceResponse :  diagramUpdateResponse.getInstanceResponse()){
						if(instanceResponse.getDeclined() != null) {
							failed.add(instanceResponse);
						}
					}
				}
			}
			
			if(!failed.isEmpty()) {				
				LogUtils.logEMF(updateResponse);
				for(ModelUpdateInstanceResponse response : failed) {
					StringBuilder message = new StringBuilder();
					if(response.getInstanceId() != null) {
						message.append(response.getInstanceId().toString());						
						if(idMap != null) {
							IIdentifier archiElement = idMap.get(response.getInstanceId());
							if(archiElement != null)
								message.append(ModelUtils.getLabel(archiElement));
						}
					}
					message.append(" [");
					message.append(response.getDeclined().toString());					
					message.append("]");
					status.add(new Status(IStatus.ERROR, CMDBPlugin.PLUGIN_ID, message.toString()));
				}
			}
		}	
		
		monitor.done();
		return status;
    }
}
