package it.unibo.cmdb.archimate;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Map.Entry;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;

import com.archimatetool.model.FolderType;
import com.archimatetool.model.IArchimateElement;
import com.archimatetool.model.IArchimateModel;
import com.archimatetool.model.IArchimatePackage;
import com.archimatetool.model.IFolder;
import com.archimatetool.model.IRelationship;
import com.archimatetool.model.util.DerivedRelationsUtils;

public class ModelClosure {
	private List<EClass> weaklist;
	IArchimateModel model;
	private Map<IArchimateElement, List<Step>> index;
	private Map<IArchimateElement, Map<IArchimateElement, EClass>> closure;
	
	private class Step {
		private IRelationship relationship;
		private boolean reverse;
		
		public Step(IRelationship relationship, boolean reverse) {
			this.relationship = relationship;
			this.reverse = reverse;
		}
		
		public EClass getType() {
			return relationship.eClass();
		}
		
		public IArchimateElement getSource() {
			return reverse ? relationship.getTarget() : relationship.getSource();
		}
		
		public IArchimateElement getTarget() {
			return reverse ? relationship.getSource() : relationship.getTarget();
		}
		
		public IRelationship getRelationship() {
			return relationship;
		}
	}

	public ModelClosure(IArchimateModel model) throws CycleDetectedException {
		weaklist = new ArrayList<EClass>();
		weaklist.add(IArchimatePackage.eINSTANCE.getAssociationRelationship());
        weaklist.add(IArchimatePackage.eINSTANCE.getAccessRelationship());
        weaklist.add(IArchimatePackage.eINSTANCE.getUsedByRelationship());
        weaklist.add(IArchimatePackage.eINSTANCE.getRealisationRelationship());
        weaklist.add(IArchimatePackage.eINSTANCE.getAssignmentRelationship());
        weaklist.add(IArchimatePackage.eINSTANCE.getAggregationRelationship());
        weaklist.add(IArchimatePackage.eINSTANCE.getCompositionRelationship());
        
        this.model = model;
        this.index = buildRelationshipsIndex(model);
        this.closure = transitiveClosure(model, index);
	}
	
	private boolean isBidirectional(EClass type) {
		return type == IArchimatePackage.eINSTANCE.getAssociationRelationship();
	}
	
	private Map<IArchimateElement, List<Step>> buildRelationshipsIndex(IArchimateModel model) {
		Map<IArchimateElement, List<Step>> index = new HashMap<IArchimateElement, List<Step>>();
		CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(model);
		
		for(IRelationship relationship : getRelationships(model))
		{
			if(!adapter.isDerived(relationship) && !adapter.isDeleted(relationship) && DerivedRelationsUtils.isStructuralRelationship(relationship)) {
				IArchimateElement source = relationship.getSource(); 
				IArchimateElement target = relationship.getTarget();
				if(source != target) {
					List<Step> sourceList = index.get(source);
					if(sourceList == null) {
						sourceList = new ArrayList<Step>();
						index.put(source, sourceList);
					}
					sourceList.add(new Step(relationship, false));
					
					List<Step> targetList = index.get(target);
					if(targetList == null) {
						targetList = new ArrayList<Step>();
						index.put(target, targetList);
					}
					if(isBidirectional(relationship.eClass()))
						targetList.add(new Step(relationship, true));
				}
			}			
		}
		return index;		
	}
	
	private Map<IArchimateElement, Map<IArchimateElement, EClass>> transitiveClosure(IArchimateModel model, Map<IArchimateElement, List<Step>> index) throws CycleDetectedException {
		Map<IArchimateElement, Map<IArchimateElement, EClass>> closure = new HashMap<IArchimateElement, Map<IArchimateElement, EClass>>();
		for(Entry<IArchimateElement, List<Step>> entry : index.entrySet()) {
			Map<IArchimateElement, EClass> pathMap = new HashMap<IArchimateElement, EClass>();
			closure.put(entry.getKey(), pathMap);
			Queue<Step> queue = new LinkedList<Step>();
			queue.addAll(entry.getValue());
			while(!queue.isEmpty()) {
				Step step = queue.remove();
				
				EClass pathType = step.getType();
				if(step.getSource() != entry.getKey())
					pathType = getWeakestType(pathType, pathMap.get(step.getSource()));
				
				if(step.getTarget() == entry.getKey()) {
					if(!isBidirectional(pathType)) {
						List<Step> cycle = findCycle(entry.getKey(), index);
						StringBuilder message = new StringBuilder();
						message.append("Cycle delected: ");
						message.append("{");
						message.append(entry.getKey().eClass().getName());
						message.append("}");
						message.append(entry.getKey().getName());
						for(Step cycleStep : cycle) {
							message.append(" --");
							message.append(cycleStep.getType().getName());
							message.append("--> ");
							message.append("{");
							message.append(cycleStep.getTarget().eClass().getName());
							message.append("}");
							message.append(cycleStep.getTarget().getName());
						}
						IStatus status = new Status(IStatus.WARNING, CMDBPlugin.PLUGIN_ID, message.toString());
						CMDBPlugin.INSTANCE.getLog().log(status);
										
						closure = null;
						throw new CycleDetectedException(message.toString());
					}
				}
				else {
					EClass oldPathType = pathMap.get(step.getTarget());
					if(oldPathType==null || getWeakestType(pathType, oldPathType)!=oldPathType) {
						pathMap.put(step.getTarget(), pathType);
						queue.addAll(index.get(step.getTarget()));
					}
				}
			}
		}		
		return closure;
	}
	
	private List<Step> findCycle(IArchimateElement element, Map<IArchimateElement, List<Step>> index) {
		List<Step> path = new LinkedList<Step>();
		return findCyclePath(element, element, path, index);
	}
	
	private List<Step> findCyclePath(IArchimateElement element, IArchimateElement current, List<Step> path, Map<IArchimateElement, List<Step>> index) {
		List<Step> cycle = null;
		Iterator<Step> iterator = index.get(current).iterator();
		while(cycle == null && iterator.hasNext()) {
			Step step = iterator.next();
			if(!path.contains(step)) {
				List<Step> list = new LinkedList<Step>(path);
				list.add(step);
				if(step.getTarget() == element)
					cycle = list;
				else
					cycle = findCyclePath(element, step.getTarget(), list, index);
			}
		}
		return cycle;
	}
		
	private EClass getWeakestType(EClass ... types) {
		int weakest = weaklist.size() - 1;
		for(EClass type : types) {
            int index = weaklist.indexOf(type);
            if(index < weakest)
                weakest = index;
        }
		return weaklist.get(weakest);
	}
	
	public EClass getPath(IArchimateElement source, IArchimateElement target) {
		EClass pathType = null;
		if(closure != null) {
			Map<IArchimateElement, EClass> entry = closure.get(source);
			if(entry != null)
				pathType = entry.get(target);
		}
		return pathType;
	}
	
	private List<IRelationship> getRelationships(IArchimateModel model) {
        List<IRelationship> relationships = new ArrayList<IRelationship>();
        IFolder folder = model.getFolder(FolderType.RELATIONS);
        getRelationships(folder, relationships);
        return relationships;
    }
    
    private void getRelationships(IFolder folder, List<IRelationship> relationships) {
        if(folder != null) {
	        for(EObject object : folder.getElements()) {
	            if(object instanceof IRelationship)
	                relationships.add((IRelationship)object);
	        }	        
	        for(IFolder f : folder.getFolders())
	        	getRelationships(f, relationships);
        }
    }
    
    public void transitiveReduction() throws CycleDetectedException {
		CMDBContentAdapter adapter = CMDBContentAdapter.getAdapter(model);
		
		for(Entry<IArchimateElement, List<Step>> indexEntry : index.entrySet()) {
			for(Step step : indexEntry.getValue()) {
				if(!isBidirectional(step.getType())) {
					boolean derived = false;
					IArchimateElement source = step.getSource();
					IArchimateElement target = step.getTarget();
					EClass type = step.getType();
					Iterator<Step> iterator1 = index.get(source).iterator();
					while(!derived && iterator1.hasNext()) {
						Step path1 = iterator1.next();
						IArchimateElement intermediate = path1.getTarget();
						if(intermediate != source && intermediate != target && getWeakestType(path1.getType(), type)==type) {
							EClass pathType2 = getPath(intermediate, target);
							if(pathType2 != null)
								derived = getWeakestType(pathType2, type)==type;
						}
					}
					if(derived) {
						IRelationship relationship = step.getRelationship();
						IFolder folder = (IFolder)relationship.eContainer();
						if(folder != null)
							folder.getElements().remove(relationship);
						adapter.getDerivedFolder().getElements().add(relationship);
					}
				}
			}
		}
	}
}