package it.unibo.cmdb.archimate;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.util.Map;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.emf.ecore.resource.impl.ExtensibleURIConverterImpl;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.m2m.atl.core.ATLCoreException;
import org.eclipse.m2m.atl.emftvm.EmftvmFactory;
import org.eclipse.m2m.atl.emftvm.ExecEnv;
import org.eclipse.m2m.atl.emftvm.Metamodel;
import org.eclipse.m2m.atl.emftvm.Model;
import org.eclipse.m2m.atl.emftvm.compiler.AtlToEmftvmCompiler;
import org.eclipse.m2m.atl.emftvm.trace.TraceLinkSet;
import org.eclipse.m2m.atl.emftvm.util.DefaultModuleResolver;
import org.eclipse.m2m.atl.emftvm.util.ModuleResolver;
import org.eclipse.m2m.atl.emftvm.util.TimingData;
import org.eclipse.m2m.atl.engine.compiler.CompileTimeError;

public class ModelTransformer {
	private ExecEnv localEnv;
	private ExecEnv remoteEnv;
	
	public class TransformationResult
	{
		private Resource resource;
		private TraceLinkSet trace;
		
		private TransformationResult(Resource resource, TraceLinkSet trace) {
			super();
			this.resource = resource;
			this.trace = trace;
		}

		public Resource getResource() {
			return resource;
		}

		public TraceLinkSet getTrace() {
			return trace;
		}
	}
	
	public ModelTransformer(String localModelName, String remoteModelName, String utilModelName, String localNs, String remoteNs, String utilNs, URI exportTransformation, URI importTransformation/*, EObject util*/) throws ATLCoreException, IOException, NoSuchAlgorithmException{
		localEnv = EmftvmFactory.eINSTANCE.createExecEnv();
		remoteEnv = EmftvmFactory.eINSTANCE.createExecEnv();
		ResourceSet rs = new ResourceSetImpl();
		
		Metamodel localMetaModel = EmftvmFactory.eINSTANCE.createMetamodel();
		localMetaModel.setResource(rs.getResource(URI.createURI(localNs), true));
		localEnv.registerMetaModel(localModelName, localMetaModel);
		remoteEnv.registerMetaModel(localModelName, localMetaModel);
		
		Metamodel remoteMetaModel = EmftvmFactory.eINSTANCE.createMetamodel();
		remoteMetaModel.setResource(rs.getResource(URI.createURI(remoteNs), true));		
		localEnv.registerMetaModel(remoteModelName, remoteMetaModel);
		remoteEnv.registerMetaModel(remoteModelName, remoteMetaModel);
		
		if(utilModelName != null && utilNs != null) {
			Metamodel utilMetaModel = EmftvmFactory.eINSTANCE.createMetamodel();
			utilMetaModel.setResource(rs.getResource(URI.createURI(utilNs), true));		
			localEnv.registerMetaModel(utilModelName, utilMetaModel);
			remoteEnv.registerMetaModel(utilModelName, utilMetaModel);			
		}
		
		IPath asmPath = CMDBPlugin.INSTANCE.getStateLocation();
		URI uriPrefix = URI.createFileURI(asmPath.toString());
		ModuleResolver resolver = new DefaultModuleResolver(uriPrefix.toString() + "/", new ResourceSetImpl());
				
		URI exportAsm = compile(asmPath, exportTransformation);
		TimingData exportTiming = new TimingData();
		localEnv.loadModule(resolver, moduleName(exportAsm));
		exportTiming.finishLoading();
		IStatus loadExportStatus = new Status(IStatus.INFO, CMDBPlugin.PLUGIN_ID,exportTransformation + " " + exportTiming.toString());
		CMDBPlugin.INSTANCE.getLog().log(loadExportStatus);
		
		URI importAsm = compile(asmPath, importTransformation);				
		TimingData importTiming = new TimingData();
		remoteEnv.loadModule(resolver, moduleName(importAsm));
		importTiming.finishLoading();
		IStatus loadImportStatus = new Status(IStatus.INFO, CMDBPlugin.PLUGIN_ID,importTransformation + " " + importTiming.toString());
		CMDBPlugin.INSTANCE.getLog().log(loadImportStatus);								
	}
	
	public void unload() {
		localEnv.clearModels();
		remoteEnv.clearModels();
	}
	
	public TransformationResult exportModel(Resource resource, EObject util) throws FileNotFoundException, ATLCoreException{
		return transform(localEnv, resource, util);
	}
	
	public TransformationResult importModel(Resource resource, EObject util) throws ATLCoreException, FileNotFoundException{
		return transform(remoteEnv, resource, util);
	}
	
	private TransformationResult transform(ExecEnv execEnv, Resource resource, EObject util) throws ATLCoreException, FileNotFoundException{

		Model inModel = EmftvmFactory.eINSTANCE.createModel();
		inModel.setResource(resource);
		execEnv.registerInOutModel("IN", inModel);
		
		Model outModel = EmftvmFactory.eINSTANCE.createModel();
		outModel.setResource(new ResourceImpl());
		execEnv.registerInOutModel("OUT", outModel);

		if(util != null) {
			Model utilModel = EmftvmFactory.eINSTANCE.createModel();
			Resource utilResource = new ResourceImpl();
			utilResource.getContents().add(util);
			utilModel.setResource(utilResource);
			execEnv.registerInputModel("UTIL", utilModel);
		}
				
		Model traceModel = EmftvmFactory.eINSTANCE.createModel();
		traceModel.setResource(new ResourceImpl());
		execEnv.registerInOutModel("trace", traceModel);
		
		TimingData timing = new TimingData();
		execEnv.run(timing);
		timing.finish();
		IStatus timingStatus = new Status(IStatus.INFO, CMDBPlugin.PLUGIN_ID, "ATL TRANFORMATION " + timing.toString());
		CMDBPlugin.INSTANCE.getLog().log(timingStatus);
		
		Resource outResource = outModel.getResource();		
		
		TraceLinkSet trace = null;
		Resource traceResource = traceModel.getResource();
		TreeIterator<EObject> iterator = traceResource.getAllContents();
		while(trace==null && iterator.hasNext()) {
			EObject eObject = iterator.next();
			if(eObject instanceof TraceLinkSet)
				trace = (TraceLinkSet) eObject;
		}
		
		execEnv.clearModels();
		
		return new TransformationResult(outResource, trace);		
	}
	
	private URI compile(IPath compiledPath, URI transformationURI) throws NoSuchAlgorithmException, UnsupportedEncodingException, IOException {
		URIConverter uriConverter = new ExtensibleURIConverterImpl();
		URI compiledURI = URI.createFileURI(compiledPath.append(moduleName(transformationURI)).addFileExtension("emftvm").toString());
		boolean compile = true;
		if(uriConverter.exists(compiledURI, null)) {
			Map<String, ?> attributes = uriConverter.getAttributes(transformationURI, null);
			Map<String, ?> compiledAttributes = uriConverter.getAttributes(compiledURI, null);
			Long lastModified = (Long)attributes.get(URIConverter.ATTRIBUTE_TIME_STAMP);
			Long compiledLastModified = (Long)compiledAttributes.get(URIConverter.ATTRIBUTE_TIME_STAMP);			
			if(compiledLastModified != null && lastModified != null && compiledLastModified > lastModified)
				compile = false;
											
		}
		if(compile) {
			LogUtils.logInfo("compiling " + transformationURI.toString() + " to " + compiledURI.toString());
			AtlToEmftvmCompiler compiler = new AtlToEmftvmCompiler();
			CompileTimeError[] errors = compiler.compile(uriConverter.createInputStream(transformationURI), compiledURI.toFileString());
			for(CompileTimeError error : errors) {
				IStatus status = new Status(IStatus.ERROR, CMDBPlugin.PLUGIN_ID, error.getSeverity() + " "
						+ error.getLocation() + " " + error.getDescription());
				CMDBPlugin.INSTANCE.getLog().log(status);				
			}
		}
		return compiledURI;
	}
	
	private String moduleName(URI uri) {
		String name = null;
		String lastSegment = uri.lastSegment();
		if (lastSegment != null) {
			int lastIndex = lastSegment.lastIndexOf('.');
			if (lastIndex == -1)
				name = lastSegment;
			else
				name = lastSegment.substring(0, lastIndex);
		}
		return name;
	}
}
