package it.unibo.cmdb.monitoring;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.emf.ecore.resource.impl.ExtensibleURIConverterImpl;
import org.eclipse.m2m.atl.core.IModel;
import org.eclipse.m2m.atl.core.IReferenceModel;
import org.eclipse.m2m.atl.core.emf.EMFInjector;
import org.eclipse.m2m.atl.core.emf.EMFModel;
import org.eclipse.m2m.atl.core.emf.EMFModelFactory;
import org.eclipse.m2m.atl.core.emf.EMFReferenceModel;
import org.eclipse.m2m.atl.core.launch.ILauncher;
import org.eclipse.m2m.atl.engine.compiler.AtlCompiler;
import org.eclipse.m2m.atl.engine.emfvm.launch.EMFVMLauncher;

public class Model2Model {
	private final static Logger LOGGER = Logger.getLogger(Model2Model.class.getName()); 
	private String inputModelName;
	private String outputModelName;
	private ILauncher transformationLauncher;
	private EMFModelFactory modelFactory;
	private EMFInjector injector;
	private IReferenceModel inputMetamodel;
	private IReferenceModel outputMetamodel;
	private Object transformationModule;
		
	public Model2Model(String compiledPath, String inputModelName, String outputModelName, String inputNs, String outputNs, URI transformation) throws Exception {
		this.inputModelName = inputModelName;
		this.outputModelName = outputModelName;
		
		transformationLauncher = new EMFVMLauncher();
		modelFactory = new EMFModelFactory();
		injector = new EMFInjector();
		
		inputMetamodel = modelFactory.newReferenceModel();
		injector.inject(inputMetamodel, inputNs);
		
		outputMetamodel = modelFactory.newReferenceModel();
		injector.inject(outputMetamodel, outputNs);
		
		URIConverter uriConverter = new ExtensibleURIConverterImpl();
		URI compitedTransformation = compile(compiledPath, transformation);
		transformationModule = transformationLauncher.loadModule(uriConverter.createInputStream(compitedTransformation));			
	}
	
	public void unload() {
		modelFactory.unload((EMFReferenceModel) inputMetamodel);
		modelFactory.unload((EMFReferenceModel) outputMetamodel);
	}
	
	public Resource transform(Resource resource){
		IModel inputModel = modelFactory.newModel(inputMetamodel);
		injector.inject(inputModel, resource);		
		IModel outputModel = modelFactory.newModel(outputMetamodel);
		
		transformationLauncher.initialize(new HashMap<String, Object>());
		transformationLauncher.addInModel(inputModel, "IN", inputModelName);
		transformationLauncher.addOutModel(outputModel, "OUT", outputModelName);
		Object error = transformationLauncher.launch(ILauncher.RUN_MODE, new NullProgressMonitor(), new HashMap<String, Object>(), transformationModule);
		if(error != null) {
			LOGGER.severe(error.toString());			
		}
		
		Resource result = ((EMFModel)outputModel).getResource();
		modelFactory.unload((EMFModel)outputModel);
		modelFactory.unload((EMFModel)inputModel);
		return result;		
	}
	
	private URI compile(String compiledPath, URI transformationURI) throws NoSuchAlgorithmException, UnsupportedEncodingException, IOException {
		URIConverter uriConverter = new ExtensibleURIConverterImpl();
		MessageDigest md = MessageDigest.getInstance("SHA-1"); 
		byte[] digest = md.digest(transformationURI.toString().getBytes("UTF-8"));
		StringBuilder signature = new StringBuilder();
		for(byte byteValue : digest)
			signature.append(String.format("%02x", byteValue));
		
		URI compiledURI = URI.createFileURI(compiledPath).appendSegment(signature.toString()).appendFileExtension("asm");
		
		boolean compile = true;
		if(uriConverter.exists(compiledURI, null)) {
			Map<String, ?> attributes = uriConverter.getAttributes(transformationURI, null);
			Map<String, ?> asmAttributes = uriConverter.getAttributes(compiledURI, null);
			Long lastModified = (Long)attributes.get(URIConverter.ATTRIBUTE_TIME_STAMP);
			Long lastModifiedAsm = (Long)asmAttributes.get(URIConverter.ATTRIBUTE_TIME_STAMP);			
			if(lastModifiedAsm != null && lastModified != null && lastModifiedAsm > lastModified)
				compile = false;
											
		}
		if(compile) {
			LOGGER.info("compiling " + transformationURI.toString() + " to " + compiledURI.toString());
			EObject[] errors = AtlCompiler.compile(uriConverter.createInputStream(transformationURI), uriConverter.createOutputStream(compiledURI));
			if(errors != null && errors.length>0) {
				for(EObject error : errors) {
					StringBuilder message = new StringBuilder();
					for(EAttribute attribute : error.eClass().getEAllAttributes()) {
						Object value = error.eGet(attribute);
						if(value !=null) {
							if(message.length()>0)
								message.append(", ");
							message.append(attribute.getName());
							message.append("=");
							message.append(value.toString());
						}
					}
					LOGGER.severe(message.toString());
				}
				new File(compiledURI.toFileString()).delete();
				compiledURI = null;
			}			
		}
		return compiledURI;
	}
}
