package it.unibo.cmdbuild.nagios;

import it.unibo.cmdbuild.rest.CMDBuildClient;
import it.unibo.cmdbuild.rest.Card;
import it.unibo.cmdbuild.rest.Domain;
import it.unibo.cmdbuild.rest.MetadataMap;
import it.unibo.cmdbuild.rest.Relation;
import it.unibo.cmdbuild.rest.Table;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Stack;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.jgrapht.alg.CycleDetector;
import org.jgrapht.graph.ListenableDirectedGraph;
import org.jgrapht.traverse.TopologicalOrderIterator;

import com.mxgraph.layout.hierarchical.mxHierarchicalLayout;
import com.mxgraph.model.mxCell;
import com.mxgraph.model.mxGeometry;
import com.mxgraph.view.mxGraph;

public class NagiosExport {

	public class NagiosExportException extends Exception {
		private static final long serialVersionUID = 1L;
		private Set<Card> cycles;

		public NagiosExportException(String message, Set<Card> cycles) {
			super(message);
			this.cycles = cycles;
		}

		public Set<Card> getCycles() {
			return cycles;
		}
	}

	private class NagiosCheckList implements Iterable<NagiosCheck> {
		private List<Card> hostPath;
		private List<NagiosCheck> checks;

		public NagiosCheckList(List<Card> hostPath) {
			this.hostPath = hostPath;
			this.checks = new ArrayList<NagiosCheck>();
		}

		public void add(Card service) {
			checks.add(new NagiosCheck(this, service));
		}

		public Iterable<Card> getHostPath() {
			return hostPath;

		}

		public Iterator<NagiosCheck> iterator() {
			return checks.iterator();
		}

		public Card getHost() {
			return hostPath.get(hostPath.size() - 1);
		}
	}

	private class NagiosCheck {
		private final NagiosCheckList checkList;
		private final Card service;

		public NagiosCheck(NagiosCheckList checkList, Card service) {
			this.service = service;
			this.checkList = checkList;
		}

		public Card getService() {
			return service;
		}

		public NagiosCheckList getCheckList() {
			return checkList;
		}
	}

	private CMDBuildClient cmdb;
	private ListenableDirectedGraph<Card, Relation> graph;
	private Map<Card, NagiosCheckList> checkMap;
	private final String HOST_TEMPLATE_CLASS;
	private final String SERVICE_CLASS;
	private final String CONTACTGROUP_CLASS;
	private final String HOST_TEMPLATE_INHERITANCE_DOMAIN;
	private final String SERVICE_ASSOCIATION_DOMAIN;
	private final String HOST_TEMPLATE_ASSOCIATION_DOMAIN;
	private final String CONTACTGROUP_ASSOCIATION_DOMAIN;
	private final String MAX_CHECK_ATTEMPTS_ATTRIBUTE;
	private final String CHECK_PERIOD_ATTRIBUTE;
	private final String NOTIFICATION_PERIOD_ATTRIBUTE;
	private final String CHECK_COMMAND_ATTRIBUTE;
	private final String CHECK_PROPAGABLE_ATTRIBUTE;
	private final String CHECK_DEPENDENCY_ATTRIBUTE;

	public NagiosExport(CMDBuildClient client) throws IOException,
			NagiosExportException {
		this.HOST_TEMPLATE_CLASS = client.getConfiguration().getProperty(
				"hostTemplateClass");
		this.SERVICE_CLASS = client.getConfiguration().getProperty(
				"serviceClass");
		this.CONTACTGROUP_CLASS = client.getConfiguration().getProperty(
				"contactGroupClass");
		this.HOST_TEMPLATE_INHERITANCE_DOMAIN = client.getConfiguration()
				.getProperty("hostTemplateInheritanceDomain");
		this.SERVICE_ASSOCIATION_DOMAIN = client.getConfiguration()
				.getProperty("serviceAssociationDomain");
		this.HOST_TEMPLATE_ASSOCIATION_DOMAIN = client.getConfiguration()
				.getProperty("hostTemplateAssociationDomain");
		this.CONTACTGROUP_ASSOCIATION_DOMAIN = client.getConfiguration()
				.getProperty("contactGroupAssociationDomain");
		this.MAX_CHECK_ATTEMPTS_ATTRIBUTE = client.getConfiguration()
				.getProperty("maxCheckAttemptsAttribute");
		this.CHECK_PERIOD_ATTRIBUTE = client.getConfiguration().getProperty(
				"checkPeriodAttribute");
		this.NOTIFICATION_PERIOD_ATTRIBUTE = client.getConfiguration()
				.getProperty("notificationPeriodAttribute");
		this.CHECK_COMMAND_ATTRIBUTE = client.getConfiguration().getProperty(
				"checkCommandAttribute");
		this.CHECK_PROPAGABLE_ATTRIBUTE = client.getConfiguration()
				.getProperty("checkPropagableAttribute");
		this.CHECK_DEPENDENCY_ATTRIBUTE = client.getConfiguration()
				.getProperty("checkDependencyAttribute");
		

		this.cmdb = client;
		graph = buildGraph();
		checkMap = buildCheckMap();
		CycleDetector<Card, Relation> cycleDetector = new CycleDetector<Card, Relation>(
				graph);
		Set<Card> cycles = cycleDetector.findCycles();
		if (!cycles.isEmpty())
			throw new NagiosExportException("Cycles detected in graph", cycles);
	}

	public Map<Card, NagiosCheckList> getCheckMap() {
		return checkMap;
	}

	public void exportNagios(Writer writer) throws IOException {
		Table hostTemplateClass = cmdb.getClassByName(HOST_TEMPLATE_CLASS);
		Table contactGroupClass = cmdb.getClassByName(CONTACTGROUP_CLASS);
		Domain hostTemplateInheritanceDomain = cmdb
				.getDomainByName(HOST_TEMPLATE_INHERITANCE_DOMAIN);
		Domain hostTemplateAssociationDomain = cmdb
				.getDomainByName(HOST_TEMPLATE_ASSOCIATION_DOMAIN);
		Domain serviceAssociationDomain = cmdb
				.getDomainByName(SERVICE_ASSOCIATION_DOMAIN);
		Domain contactgroupAssociationDomain = cmdb
				.getDomainByName(CONTACTGROUP_ASSOCIATION_DOMAIN);
		Set<Card> hostGroupSet = new HashSet<Card>();
		
		for (Card template : hostTemplateClass.cards()) {
			// HOST template
			writer.write("define host{\n");
			writer.write("\tname host_" + name(template) + "\n");
			writer.write("\thostgroups " + name(template) + "\n");
			boolean hasHostContacts = false;
			for (Relation relation : template
					.relations(Relation.Direction.DIRECT)) {
				if (relation.getSchema() == contactgroupAssociationDomain) {
					Card target = relation.getCard2();
					assert(target.getSchema() == contactGroupClass);
					if(!hasHostContacts) {
						hasHostContacts = true;
						writer.write("\tcontact_groups ");
					}
					else 
						writer.write(",");
					writer.write(name(target));
				}
			}
			if(hasHostContacts)
				writer.write("\n");
			if (template.hasAttribute(MAX_CHECK_ATTEMPTS_ATTRIBUTE))
				writer.write("\tmax_check_attempts "
						+ template.get(MAX_CHECK_ATTEMPTS_ATTRIBUTE).getValue()
						+ "\n");
			if (template.hasAttribute(CHECK_PERIOD_ATTRIBUTE))
				writer.write("\tcheck_period "
						+ template.get(CHECK_PERIOD_ATTRIBUTE).getValue()
						+ "\n");
			if (template.hasAttribute(NOTIFICATION_PERIOD_ATTRIBUTE))
				writer.write("\tnotification_period "
						+ template.get(NOTIFICATION_PERIOD_ATTRIBUTE)
								.getValue() + "\n");
			boolean hasHostTemplate = false;
			for (Relation relation : template
					.relations(Relation.Direction.DIRECT)) {
				if (relation.getSchema() == hostTemplateInheritanceDomain) {
					Card target = relation.getCard2();
					assert (!hasHostTemplate);
					assert (relation.getCard2().getSchema() == hostTemplateClass);
					if(!hasHostTemplate) {
						hasHostTemplate = true;
						writer.write("\tuse host_" + name(target) + "\n");
					}
				}
			}
			if(!hasHostTemplate)
				writer.write("\tuse generic-host\n");
			writer.write("\tnotes_url /nagiosbp/cgi-bin/whereUsed.cgi?host=$HOSTNAME$\n");
			writer.write("\tregister 0\n");
			writer.write("}\n\n");

			// SERVICE template
			writer.write("define service{\n");
			writer.write("\tname service_" + name(template) + "\n");
			boolean hasServiceContacts = false;
			for (Relation relation : template
					.relations(Relation.Direction.DIRECT)) {
				if (relation.getSchema() == contactgroupAssociationDomain) {
					Card target = relation.getCard2();
					assert(target.getSchema() == contactGroupClass);
					if(!hasServiceContacts) {
						hasServiceContacts = true;
						writer.write("\tcontact_groups ");
					}
					else 
						writer.write(",");
					writer.write(name(target));
				}
			}
			if(hasServiceContacts)
				writer.write("\n");
			if (template.hasAttribute(MAX_CHECK_ATTEMPTS_ATTRIBUTE))
				writer.write("\tmax_check_attempts "
						+ template.get(MAX_CHECK_ATTEMPTS_ATTRIBUTE).getValue()
						+ "\n");
			if (template.hasAttribute(CHECK_PERIOD_ATTRIBUTE))
				writer.write("\tcheck_period "
						+ template.get(CHECK_PERIOD_ATTRIBUTE).getValue()
						+ "\n");
			if (template.hasAttribute(NOTIFICATION_PERIOD_ATTRIBUTE))
				writer.write("\tnotification_period "
						+ template.get(NOTIFICATION_PERIOD_ATTRIBUTE)
								.getValue() + "\n");
			boolean hasServiceTemplate = false;
			for (Relation relation : template
					.relations(Relation.Direction.DIRECT)) {
				if (relation.getSchema() == hostTemplateInheritanceDomain) {
					Card target = relation.getCard2();
					assert (!hasServiceTemplate);
					assert (relation.getCard2().getSchema() == hostTemplateClass);
					if(!hasServiceTemplate) {
						hasServiceTemplate = true;
						writer.write("\tuse service_" + name(target) + "\n");
					}
				}
			}
			if(!hasServiceTemplate)
				writer.write("\tuse generic-service\n");
			writer.write("\tnotes_url /nagiosbp/cgi-bin/whereUsed.cgi?host=$HOSTNAME$&service=$SERVICEDESC$\n");
			writer.write("\tregister 0\n");
			writer.write("}\n\n");
		}

		for (Card card : graph.vertexSet()) {
			Map<String, List<Relation>> relationClusters = clusterizeRelations(graph.outgoingEdgesOf(card));
			NagiosCheckList checkList = checkMap.get(card);
			
			Card template = null;
			if(checkList != null) {
				for(Card current : checkList.hostPath) {
					if(template == null) {
						for (Relation relation : current.relations(Relation.Direction.DIRECT)) {
							if (relation.getSchema() == hostTemplateAssociationDomain) {
								assert (template == null);
								assert (relation.getCard2().getSchema() == hostTemplateClass);
								template = relation.getCard2();
							}
						}
					}
				}
			}

			// HOST
			if (isHost(card)) {
				writer.write("define host{\n");
				writer.write("\thost_name " + name(card) + "\n");
				writer.write("\tdisplay_name " + description(card) + "\n");
				writer.write("\taddress " + hostAddress(card) + "\n");
				writer.write("\tuse host_" + name(template) + "\n");
				writer.write("\thostgroups " + name(template) + "\n");
				writer.write("}\n\n");
				hostGroupSet.add(template);
			}

			// SERVICEGROUP
			if (isServiceGroup(card)) {
				writer.write("define servicegroup{\n");
				writer.write("\tservicegroup_name " + name(card) + "\n");
				writer.write("\talias " + description(card) + "\n");
				/*boolean hasMembers = false;	
				for(Relation relation : graph.outgoingEdgesOf(card)) {
					if(isCheckAssociation(relation)) {
						Card target = graph.getEdgeTarget(relation);
						if(isServiceGroup(target)) {
							if(!hasMembers) {
								hasMembers = true;
								writer.write("\tservicegroup_members ");
							}
							else 
								writer.write(",");
							writer.write(name(target));
						}
					}
				}
				if(hasMembers)
					writer.write("\n");*/
				writer.write("}\n\n");
			}
			
			if(checkList != null)
			{
				// cluster SERVICE for card
				writer.write("define service{\n");
				writer.write("\thost_name " + name(checkList.getHost()) + "\n");
				writer.write("\tservice_description " + name(card) + "\n");
				writer.write("\tdisplay_name " + description(card) + "\n");
				writer.write("\tcheck_command check_service_cluster!\""	+ description(card) + "\"!1!1!0");
				for (NagiosCheck check : checkList) {
					Card service = check.getService();
					if (service.hasAttribute(CHECK_DEPENDENCY_ATTRIBUTE) && service.get(CHECK_DEPENDENCY_ATTRIBUTE).getBoolean())
						writer.write(",$SERVICESTATEID:" + name(checkList.getHost()) + ":" + serviceName(check) + "$");
				}
				writer.write("\n");
				writer.write("\tuse service_" + name(template) + "\n");
				writer.write("}\n\n");
	
				// cluster SERVICE
				for (Entry<String, List<Relation>> entry : relationClusters
						.entrySet()) {
					if (!entry.getKey().isEmpty()) {
						writer.write("define service{\n");
						writer.write("\thost_name " + name(checkList.getHost()) + "\n");
						writer.write("\tservice_description " + clusterName(card, entry.getKey()) + "\n");
						writer.write("\tdisplay_name " + clusterName(card, entry.getKey()) + "\n");
						writer.write("\tcheck_command check_service_cluster!\""	+ clusterName(card, entry.getKey()) + "\"");
						int count = entry.getValue().size();
						writer.write("!");
						writer.write(Integer.toString(1));
						writer.write("!");
						writer.write(Integer.toString(count - 1));
						writer.write("!");
						boolean first = true;
						for (Relation relation : entry.getValue()) {
							if (first)
								first = false;
							else
								writer.write(",");
							if (isCheckDependency(relation)) {
								Card target = graph.getEdgeTarget(relation);
								NagiosCheckList targetCheckList = checkMap
										.get(target);
								writer.write("$SERVICESTATEID:"
										+ name(targetCheckList.getHost()) + ":"
										+ name(target) + "$");
							} else
								writer.write("0");
						}
						writer.write("\n");
						writer.write("\tuse service_" + name(template) + "\n");
						writer.write("}\n\n");
					}
				}
	
				for (NagiosCheck check : checkList) {
					Card checkTemplate = null;
					for (Relation relation : check.getService().relations(
							Relation.Direction.INVERSE)) {
						if (relation.getSchema() == serviceAssociationDomain) {
							assert (checkTemplate == null);
							assert (relation.getCard1().getSchema() == hostTemplateClass);
							checkTemplate = relation.getCard1();
						}
					}
	
					// SERVICE
					writer.write("define service{\n");
					writer.write("\thost_name " + name(checkList.getHost()) + "\n");
					writer.write("\tservice_description " + serviceName(check)
							+ "\n");
					writer.write("\tdisplay_name "
							+ description(check.getService()) + "\n");
					writer.write("\tcheck_command " + command(check) + "\n");
					writer.write("\tservicegroups " + name(card) + "\n");
					writer.write("\tuse service_" + name(checkTemplate) + "\n");
					writer.write("}\n\n");
	
					for (Entry<String, List<Relation>> entry : relationClusters
							.entrySet()) {
						if (!entry.getKey().isEmpty()) {
							// cluster SERVICEDEPENDENCY
							writer.write("define servicedependency{\n");
							writer.write("\tdependent_host_name "
									+ name(checkList.getHost()) + "\n");
							writer.write("\tdependent_service_description "
									+ serviceName(check) + "\n");
							writer.write("\thost_name " + name(checkList.getHost())
									+ "\n");
							writer.write("\tservice_description "
									+ clusterName(card, entry.getKey()) + "\n");
							writer.write("\texecution_failure_criteria n\n");
							writer.write("\tnotification_failure_criteria w,u,c\n");
							writer.write("}\n\n");
						} else {
							for (Relation relation : entry.getValue()) {
								// SERVICEDEPENDENCY
								if (isCheckDependency(relation)) {
									Card target = graph.getEdgeTarget(relation);
									NagiosCheckList targetCheckList = checkMap
											.get(target);
									if(targetCheckList != null) {
										writer.write("define servicedependency{\n");
										writer.write("\tdependent_host_name "
												+ name(checkList.getHost()) + "\n");
										writer.write("\tdependent_service_description "
												+ serviceName(check) + "\n");
										writer.write("\thost_name "
												+ name(targetCheckList.getHost()) + "\n");
										writer.write("\tservice_description "
												+ name(target) + "\n");
										writer.write("\texecution_failure_criteria n\n");
										writer.write("\tnotification_failure_criteria w,u,c\n");
										writer.write("}\n\n");
									}
								}
							}
						}
					}
				}
			}
		}
		
		Stack<Card> hostGroupStack = new Stack<Card>();
		hostGroupStack.addAll(hostGroupSet);
		
		while(!hostGroupStack.empty()) {
			Card template = hostGroupStack.pop();
			for (Relation relation : template.relations(Relation.Direction.DIRECT)) {
				if (relation.getSchema() == hostTemplateInheritanceDomain) {
					Card parent = relation.getCard2();
					if(hostGroupSet.add(parent))
						hostGroupStack.push(parent);
				}
			}
		}
		
		for(Card template : hostGroupSet) {
			// HOSTGROUP
			writer.write("define hostgroup{\n");
			writer.write("\thostgroup_name " + name(template) + "\n");
			writer.write("\talias " + description(template) + "\n");
			boolean hasMembers = false;
			for (Relation relation : template.relations(Relation.Direction.INVERSE)) {
				if (relation.getSchema() == hostTemplateInheritanceDomain) {
					Card target = relation.getCard1();
					assert (target.getSchema() == hostTemplateClass);
					if(hostGroupSet.contains(target)) {
						if(!hasMembers) {
							hasMembers = true;
							writer.write("\thostgroup_members ");
						}
						else 
							writer.write(",");
						writer.write(name(target));
					}
				}
			}
			if(hasMembers)
				writer.write("\n");
			writer.write("}\n\n");
		}
	}

	public void exportBP(Writer writer) throws IOException {
		Stack<Card> stack = new Stack<Card>();
		TopologicalOrderIterator<Card, Relation> iterator = new TopologicalOrderIterator<Card, Relation>(
				graph);
		while (iterator.hasNext())
			stack.push(iterator.next());
		while (!stack.empty()) {
			Card card = stack.pop();

			Map<String, List<Relation>> relationClusters = clusterizeRelations(graph
					.outgoingEdgesOf(card));
			for (Entry<String, List<Relation>> entry : relationClusters
					.entrySet()) {
				if (!entry.getKey().isEmpty()) {
					writer.write(bpName(clusterName(card, entry.getKey())));
					writer.write(" = ");
					boolean first = true;
					for (Relation relation : entry.getValue()) {
						if (first)
							first = false;
						else
							writer.write(" | ");
						writer.write(bpName(name(graph.getEdgeTarget(relation))));
					}
					writer.write("\n");
					writer.write("display 0;");
					writer.write(bpName(clusterName(card, entry.getKey())));
					writer.write(";");
					writer.write(description(card));
					writer.write(" - CLUSTER ");
					writer.write(entry.getKey());
					writer.write("\n");
				}
			}

			writer.write(bpName(name(card)));
			writer.write(" = ");
			boolean first = true;
			for (Entry<String, List<Relation>> entry : relationClusters
					.entrySet()) {
				if (!entry.getKey().isEmpty()) {
					if (first)
						first = false;
					else
						writer.write(" & ");
					writer.write(bpName(clusterName(card, entry.getKey())));
				} else {
					for (Relation relation : entry.getValue()) {
						if (first)
							first = false;
						else
							writer.write(" & ");
						writer.write(bpName(name(graph.getEdgeTarget(relation))));
					}
				}
			}
			NagiosCheckList checkList = checkMap.get(card);
			if (checkList != null) {
				for (NagiosCheck check : checkList) {
					if (first)
						first = false;
					else
						writer.write(" & ");
					writer.write(name(checkList.getHost()));
					writer.write(";");
					writer.write(serviceName(check));
				}
			}
			writer.write("\n");
			writer.write("display ");
			writer.write(Integer.toString(priority(card)));
			writer.write(";");
			writer.write(bpName(name(card)));
			writer.write(";");
			writer.write(description(card));
			writer.write("\n");
			if(isMap(card)) {
				writer.write("info_url ");
				writer.write(bpName(name(card)));
				writer.write(";");
				writer.write("/nagvis/frontend/nagvis-js/index.php?mod=Map&act=view&show=");
				writer.write(card.getSchema().getName());
				writer.write(Integer.toString(card.getId()));
				writer.write("#\n");
			}
		}
	}
	
	public mxGraph exportNagvis(Card root, Writer writer) throws IOException{
		Set<Card> vertexSet = new HashSet<Card>();
		Set<Relation> edgeSet = new HashSet<Relation>();
		Set<Card> dependencySet = new HashSet<Card>();
		Stack<Card> stack = new Stack<Card>();
		stack.push(root);
		while(!stack.empty()) {
			Card current = stack.pop();
			if(vertexSet.add(current)) {
				for(Relation relation : graph.outgoingEdgesOf(current)) {
					edgeSet.add(relation);
					Card target = graph.getEdgeTarget(relation);
					if(isInternalDependency(relation))
						stack.push(target);
					else
						dependencySet.add(target);
				}
			}
		}
		vertexSet.addAll(dependencySet);
		
		mxGraph graphX = new mxGraph();
		graphX.getModel().beginUpdate();
		try {
			HashMap<Card, mxCell> vertexToCellMap = new HashMap<Card, mxCell>();
			for (Card vertex : vertexSet) {
				mxCell cell = new mxCell(vertex);
				cell.setVertex(true);
				cell.setId(null);
				cell.setGeometry(new mxGeometry(0, 0, 100, 20));
				graphX.addCell(cell, graphX.getDefaultParent());
				vertexToCellMap.put(vertex, cell);
			}
			for (Relation edge : edgeSet) {
				Card source = graph.getEdgeSource(edge);
				Card target = graph.getEdgeTarget(edge);
				mxCell cell = new mxCell(edge);
				cell.setEdge(true);
				cell.setId(null);
				cell.setGeometry(new mxGeometry());
				cell.getGeometry().setRelative(true);
				graphX.addEdge(cell, graphX.getDefaultParent(), vertexToCellMap.get(source), vertexToCellMap.get(target), null);
			}
		} finally {
			graphX.getModel().endUpdate();
		}
		mxHierarchicalLayout layout = new mxHierarchicalLayout(graphX);
		layout.setInterRankCellSpacing(100);
		layout.execute(graphX.getDefaultParent());
		
		writer.write("define global {\n");
		writer.write("\talias=" + description(root) + "\n");
		writer.write("\tbackend_id=nagiosbp_1\n");
		writer.write("\tlabel_show=1\n");
		writer.write("\ticonset=std_medium\n");
		writer.write("}\n\n");
			
		for(Object child : graphX.getChildVertices(graphX.getDefaultParent())) {
			mxCell cell = (mxCell)child;
			Card card = (Card)cell.getValue();
			writer.write("define servicegroup {\n");
			writer.write("\tservicegroup_name=" + bpName(name(card)) + "\n");
			writer.write("\tx=");
			writer.write(Long.toString(Math.round(cell.getGeometry().getX())));
			writer.write("\n");	
			writer.write("\ty=");
			writer.write(Long.toString(Math.round(cell.getGeometry().getY())));
			writer.write("\n");	
			writer.write("\tz=1\n");	
			writer.write("\tlabel_text=" + description(card) + "\n");
			writer.write("\tlabel_style=font-size:xx-small\n");
			writer.write("\tlabel_border=#FFFFFF\n");
			writer.write("\tlabel_background=#FFFFFF\n");
			writer.write("}\n\n");
		}
		
		for(Object child : graphX.getChildEdges(graphX.getDefaultParent())) {
			mxCell cell = (mxCell)child;
			mxCell source = (mxCell)cell.getSource();
			mxCell target = (mxCell)cell.getTarget();
			double length = Math.sqrt(Math.pow(source.getGeometry().getX() - target.getGeometry().getX(), 2) +
					Math.pow(source.getGeometry().getY() - target.getGeometry().getY(), 2));
			double dx = (target.getGeometry().getX() - source.getGeometry().getX()) / length;
			double dy = (target.getGeometry().getY() - source.getGeometry().getY()) / length;
			double radius = 10;
			writer.write("define servicegroup {\n");
			writer.write("\tservicegroup_name=" + bpName(name((Card)target.getValue())) + "\n");
			writer.write("\tline_type=11\n");
			writer.write("\tx=");
			writer.write(Long.toString(Math.round(source.getGeometry().getX() + dx*radius)));
			writer.write(",");						
			writer.write(Long.toString(Math.round(target.getGeometry().getX() - dx*radius)));
			writer.write("\n");	
			writer.write("\ty=");
			writer.write(Long.toString(Math.round(source.getGeometry().getY() + dy*radius)));
			writer.write(",");						
			writer.write(Long.toString(Math.round(target.getGeometry().getY() - dy*radius)));
			writer.write("\n");
			writer.write("\tz=0\n");
			writer.write("\tlabel_text=" + ((Relation)cell.getValue()).getSchema().getName() + "\n");
			writer.write("\tline_cut=0.5\n");
			writer.write("\tline_width=2\n");
			writer.write("\tview_type=line\n");
			writer.write("\tlabel_x=");
			double labelDx = (target.getGeometry().getX() - source.getGeometry().getX()) / 2 - 15;
			if(labelDx >= 0)
				writer.write("+");
			writer.write(Long.toString(Math.round(labelDx)));
			writer.write("\n");
			writer.write("\tlabel_y=");
			double labelDy = (target.getGeometry().getY() - source.getGeometry().getY()) / 2 + 10;
			if(labelDy >= 0)
				writer.write("+");
			writer.write(Long.toString(Math.round(labelDy)));
			writer.write("\n");
			writer.write("\tlabel_style=font-size:xx-small;color:#AAAAAA\n");
			writer.write("\tlabel_border=#FFFFFF\n");
			//writer.write("\tlabel_background=#FFFFFF\n");
			writer.write("\tlabel_show=1\n");
			writer.write("\thover_menu=0\n");
			writer.write("}\n\n");
		}
		return graphX;
	}

	private ListenableDirectedGraph<Card, Relation> buildGraph() {
		ListenableDirectedGraph<Card, Relation> graph = new ListenableDirectedGraph<Card, Relation>(
				Relation.class);
		for (Table clazz : cmdb.classes()) {
			if (!clazz.isSuperClass() && clazz.getType() == Table.Type.CLASS) {
				MetadataMap metadata = clazz.getMetadata();
				if (Boolean.parseBoolean(metadata.get("host"))
						|| Boolean.parseBoolean(metadata.get("serviceGroup"))) {
					for (Card card : clazz.cards())
						graph.addVertex(card);
				}
			}
		}
		for (Card card : graph.vertexSet()) {
			for (Relation relation : card.relations(Relation.Direction.DIRECT)) {
				if (getDirection(relation) == Relation.Direction.DIRECT) {
					if (isCheckAssociation(relation)
							|| isCheckDependency(relation)
							|| isCkeckPropagation(relation)) {
						Card target = relation.getCard2();
						if (graph.containsVertex(target))
							graph.addEdge(card, target, relation);
					}
				}
			}
			for (Relation relation : card.relations(Relation.Direction.INVERSE)) {
				if (getDirection(relation) == Relation.Direction.INVERSE) {
					if (isCheckAssociation(relation)
							|| isCheckDependency(relation)
							|| isCkeckPropagation(relation)) {
						Card target = relation.getCard1();
						if (graph.containsVertex(target))
							graph.addEdge(card, target, relation);
					}
				}
			}
		}
		return graph;
	}

	private Map<String, List<Relation>> clusterizeRelations(
			Collection<Relation> relations) {
		Map<String, List<Relation>> relationClusters = new HashMap<String, List<Relation>>();
		for (Relation relation : relations) {
			MetadataMap metadata = relation.getSchema().getMetadata();
			String clusterAttribute = metadata.get("clusterAttribute");
			String cluster = relation.hasAttribute(clusterAttribute) ? relation
					.get(clusterAttribute).getValue() : "";
			if (!relationClusters.containsKey(cluster))
				relationClusters.put(cluster, new ArrayList<Relation>());
			relationClusters.get(cluster).add(relation);
		}
		return relationClusters;
	}

	private Map<Card, NagiosCheckList> buildCheckMap() {
		Table serviceClass = cmdb.getClassByName(SERVICE_CLASS);
		Table hostTemplateClass = cmdb.getClassByName(HOST_TEMPLATE_CLASS);
		Domain hostTemplateInheritanceDomain = cmdb
				.getDomainByName(HOST_TEMPLATE_INHERITANCE_DOMAIN);
		Domain hostTemplateAssociationDomain = cmdb
				.getDomainByName(HOST_TEMPLATE_ASSOCIATION_DOMAIN);
		Domain serviceAssociationDomain = cmdb
				.getDomainByName(SERVICE_ASSOCIATION_DOMAIN);
		Map<Card, NagiosCheckList> checks = new HashMap<Card, NagiosCheckList>();

		Stack<Card> stack = new Stack<Card>();
		TopologicalOrderIterator<Card, Relation> iterator = new TopologicalOrderIterator<Card, Relation>(
				graph);
		while (iterator.hasNext())
			stack.push(iterator.next());
		while (!stack.empty()) {
			Card card = stack.pop();
			
			// host path
			List<Card> hostPath = new ArrayList<Card>();
			hostPath.add(card);
			Card current = card;
			while (current != null && !isHost(current)) {
				boolean found = false;
				for (Relation relation : graph.outgoingEdgesOf(current)) {
					if (isCheckAssociation(relation)) {
						assert (!found);
						if (!found) {
							hostPath.add(graph.getEdgeTarget(relation));
							found = true;
						}
					}
				}
				current = found ? hostPath.get(hostPath.size() - 1) : null;
			}
			assert (current != null);
			if (current != null)
				checks.put(card, new NagiosCheckList(hostPath));

			// associated check
			for (Relation relation : card.relations(Relation.Direction.DIRECT)) {
				if (relation.getSchema() == hostTemplateAssociationDomain) {
					Card template = relation.getCard2();
					while (template != null) {
						assert (template.getSchema() == hostTemplateClass);
						Card parent = null;
						for (Relation templateRelation : template
								.relations(Relation.Direction.DIRECT)) {
							if (templateRelation.getSchema() == serviceAssociationDomain) {
								Card service = templateRelation.getCard2();
								assert (service.getSchema() == serviceClass);
								checks.get(card).add(service);
							} else if (templateRelation.getSchema() == hostTemplateInheritanceDomain) {
								Card parentTemplate = templateRelation
										.getCard2();
								assert (parentTemplate.getSchema() == serviceClass);
								assert (parent == null);
								parent = parentTemplate;
							}
						}
						template = parent;
					}
				}
			}
			checkPropagation(card, checks);
		}
		return checks;
	}

	private void checkPropagation(Card card, Map<Card, NagiosCheckList> checks) {
		NagiosCheckList checkList = checks.get(card);
		for (Relation relation : card.relations(Relation.Direction.DIRECT)) {
			if (isCkeckPropagation(relation)) {
				Card target = relation.getCard2();
				NagiosCheckList targetCheckList = checks.get(target);
				assert (targetCheckList != null);
				if (targetCheckList != null) {
					for (NagiosCheck check : checkList) {
						Card service = check.getService();
						if (service.hasAttribute(CHECK_PROPAGABLE_ATTRIBUTE)
								&& service.get(CHECK_PROPAGABLE_ATTRIBUTE)
										.getBoolean())
						targetCheckList.add(service);
					}
				}
				checkPropagation(target, checks);
			}
		}
	}

	private String name(Card card) {
		MetadataMap metadata = card.getSchema().getMetadata();
		String attribute = metadata.get("nameAttribute");
		return card.hasAttribute(attribute) ? card.get(attribute).getValue()
				: "";
	}

	private String description(Card card) {
		MetadataMap metadata = card.getSchema().getMetadata();
		String attribute = metadata.get("descriptionAttribute");
		return card.hasAttribute(attribute) ? card.get(attribute).getValue()
				: "";
	}

	private int priority(Card card) {
		MetadataMap metadata = card.getSchema().getMetadata();
		String attribute = metadata.get("priorityAttribute");
		return card.hasAttribute(attribute) ? card.get(attribute).getInt() : 0;
	}

	private String hostAddress(Card card) {
		MetadataMap metadata = card.getSchema().getMetadata();
		String attribute = metadata.get("hostAddressAttribute");
		return card.hasAttribute(attribute) ? card.get(attribute).getValue()
				: "";
	}

	public boolean isHost(Card card) {
		MetadataMap metadata = card.getSchema().getMetadata();
		return Boolean.parseBoolean(metadata.get("host"));
	}

	public boolean isServiceGroup(Card card) {
		MetadataMap metadata = card.getSchema().getMetadata();
		return Boolean.parseBoolean(metadata.get("serviceGroup"));
	}

	public boolean isMap(Card card) {
		MetadataMap metadata = card.getSchema().getMetadata();
		return Boolean.parseBoolean(metadata.get("map"));
	}

	public boolean isCheckDependency(Relation relation) {
		MetadataMap metadata = relation.getSchema().getMetadata();
		return Boolean.parseBoolean(metadata.get("ckeckDependency"));
	}

	public boolean isCkeckPropagation(Relation relation) {
		MetadataMap metadata = relation.getSchema().getMetadata();
		return Boolean.parseBoolean(metadata.get("ckeckPropagation"));
	}

	private boolean isCheckAssociation(Relation relation) {
		MetadataMap metadata = relation.getSchema().getMetadata();
		return Boolean.parseBoolean(metadata.get("checkAssociation"));
	}

	public boolean isInternalDependency(Relation relation) {
		MetadataMap metadata = relation.getSchema().getMetadata();
		return Boolean.parseBoolean(metadata.get("internalDependency"));
	}

	private Relation.Direction getDirection(Relation relation) {
		MetadataMap metadata = relation.getSchema().getMetadata();
		String direction = (String) metadata.get("direction");
		return direction != null ? Enum.valueOf(Relation.Direction.class,
				direction) : Relation.Direction.DIRECT;
	}

	private String clusterName(Card card, String cluster) {
		return name(card) + "_" + cluster;
	}

	private String serviceName(NagiosCheck check) {
		StringBuffer buffer = new StringBuffer();
		for (Card card : check.getCheckList().getHostPath()) {
			buffer.append(name(card));
			buffer.append("_");
		}
		buffer.append(name(check.service));
		return buffer.toString();
	}
	
	private String bpName(String name){
		return name.replaceAll("[^A-Za-z0-9_-]", "-");
	}
	
	private String command(NagiosCheck check) {
		Card service = check.getService();
		String text = service.get(CHECK_COMMAND_ATTRIBUTE).getValue();
		Pattern pattern = Pattern.compile("\\[(.+?)\\]");
		Matcher matcher = pattern.matcher(text);
		StringBuilder builder = new StringBuilder();
		int i = 0;
		while (matcher.find()) {
			String name = matcher.group(1);
			String replacement = null;
			Iterator<Card> iterator = check.getCheckList().getHostPath()
					.iterator();
			while (replacement == null && iterator.hasNext()) {
				Card card = iterator.next();
				if (card.hasAttribute(name))
					replacement = card.get(name).getValue();
			}
			builder.append(text.substring(i, matcher.start()));
			if (replacement != null)
				builder.append(replacement);
			//else
			//	builder.append(matcher.group(0));
			i = matcher.end();
		}
		builder.append(text.substring(i, text.length()));
		return builder.toString();
	}
}